/*
 * Copyright 2022 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.seppiko.commons.utils.jdbc;

import java.sql.Date;
import java.sql.Time;
import java.sql.Timestamp;
import java.time.LocalDateTime;
import org.seppiko.commons.utils.ObjectUtil;

/**
 * Sql Type Util
 *
 * @author Leonard Woo
 */
public class SqlTypeUtil {

  /**
   * Convert SQL datetime to Local datetime
   *
   * @param date SQL date
   * @param time SQL time
   * @return Local datetime
   */
  public static LocalDateTime convertSqlDateTime(Date date, Time time) {
    return date.toLocalDate().atTime(time.toLocalTime());
  }

  /**
   * Convert SQL Timestamp with zone to Local DateTime
   *
   * @param timestamp SQL Timestamp
   * @return Local DateTime
   */
  public static LocalDateTime convertSqlTimestamp(Timestamp timestamp) {
    return timestamp.toLocalDateTime();
  }

  /**
   * Convert Sql Object to Class type
   *
   * @param sqlObj sql object
   * @param type target class type
   * @param <T> type
   * @return Class type object
   * @throws NullPointerException sql object is null
   * @throws IllegalArgumentException target type mismatch
   */
  @SuppressWarnings("unchecked")
  public static <T> T convertTypeObject(Object sqlObj, Class<T> type)
      throws NullPointerException, IllegalArgumentException {
    if (ObjectUtil.isNull(sqlObj)) {
      throw new NullPointerException();
    }
    if (sqlObj.getClass().equals(type)) {
      return (T) sqlObj;
    }
    throw new IllegalArgumentException();
  }

}
