/*
 * Copyright 2022 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.seppiko.commons.utils;

import java.nio.ByteBuffer;
import java.nio.CharBuffer;
import java.nio.charset.CharacterCodingException;
import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;

/**
 * Character Util
 *
 * @author Leonard Woo
 */
public class CharUtil {

  /** NUL or {@code '\0'} */
  public static final Character NULL = 0x00;

  /** HT or {@code '\t'} */
  public static final Character HORIZONTAL_TABULATION = 0x09;

  /** LF or {@code '\n'} */
  public static final Character LINE_FEED = 0x0A;

  /** VT or {@code '\v'} */
  public static final Character VERTICAL_TABULATION = 0x0B;

  /** FF or {@code '\f'} */
  public static final Character FORM_FEED = 0x0C;

  /** CR or {@code '\r'} */
  public static final Character CARRIAGE_RETURN = 0x0D;

  /** CRLF or {@code "\r\n"} */
  public static final String CRLF = String.valueOf(new char[] {CARRIAGE_RETURN, LINE_FEED});

  /** {@code '.'} */
  public static final Character FULL_STOP = 0x2E;

  /** {@code '-'} */
  public static final Character HYPHEN_MINUS = 0xFF0D;

  /** {@code '+'} */
  public static final Character PLUS = 0x2B;

  /** {@code '*'} */
  public static final Character ASTERISK = 0x2A;

  /** {@code '/'} */
  public static final Character SOLIDUS = 0x2F;

  /** {@code '\'} */
  public static final Character REVERSE_SOLIDUS = 0x5C;

  /** {@code '．'} */
  public static final Character FULLWIDTH_FULL_STOP = 0xFF0E;

  /** {@code '－'} */
  public static final Character FULLWIDTH_HYPHEN_MINUS = 0xFF0D;

  /** {@code '＋'} */
  public static final Character FULLWIDTH_PLUS = 0xFF0B;

  /** {@code '＊'} */
  public static final Character FULLWIDTH_ASTERISK = 0xFF0A;

  /** {@code '／'} */
  public static final Character FULLWIDTH_SOLIDUS = 0xFF0F;

  /** {@code '＼'} */
  public static final Character FULLWIDTH_REVERSE_SOLIDUS = 0xFF3C;

  /**
   * Charset decode
   *
   * @param charset charset, e.g. {@code StandardCharsets.UTF_8}
   * @param data byte array
   * @return character buffer object
   * @see Charset
   * @see StandardCharsets
   */
  public static CharBuffer charsetDecode(Charset charset, byte[] data) {
    try {
      return charset.newDecoder().decode(ByteBuffer.wrap(data));
    } catch (RuntimeException | CharacterCodingException ignored) {
    }
    return null;
  }

  /**
   * Charset encode
   *
   * @param charset charset, e.g. {@code StandardCharsets.UTF_8}
   * @param data character buffer
   * @return byte array
   * @see Charset
   * @see StandardCharsets
   */
  public static byte[] charsetEncode(Charset charset, CharBuffer data) {
    try {
      return charset.newEncoder().encode(data).array();
    } catch (RuntimeException | CharacterCodingException ignored) {
    }
    return null;
  }

  /**
   * Determines if the specified character is a digit.
   *
   * <p>Some Unicode character ranges that contain digits:
   *
   * <ul>
   *   <li>{@code '\u005Cu0030'} through {@code '\u005Cu0039'}, ISO-LATIN-1 digits ({@code '0'}
   *       through {@code '9'})
   *   <li>{@code '\u005CuFF10'} through {@code '\u005CuFF19'}, Fullwidth digits ({@code '０'}
   *       through {@code '９'})
   * </ul>
   *
   * @param ch the character to be tested.
   * @return true, if the character is a digit; false, otherwise.
   */
  public static boolean isDigit(char ch) {
    return NumberUtil.between(ch, 0x30, 0x39) || NumberUtil.between(ch, 0xFF10, 0xFF19);
  }

  /**
   * Determines if the specified character is a full stop.
   *
   * <ul>
   *   <li>{@code '\u005Cu002E'}, ISO-LATIN-1 full stop ({@code '.'}
   *   <li>{@code '\u005CuFF0E'}, Fullwidth digits ({@code '．'}
   * </ul>
   *
   * @param ch the character to be tested.
   * @return true, if the character is a digit; false, otherwise.
   */
  public static boolean isFullStop(char ch) {
    return ch == FULL_STOP || ch == FULLWIDTH_FULL_STOP;
  }
}
