/*
 * Copyright 2022 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.seppiko.commons.utils.codec;

import org.seppiko.commons.utils.MathUtil;
import org.seppiko.commons.utils.NumberUtil;

/**
 * Base62 digit
 *
 * @see <a href="https://en.wikipedia.org/wiki/Base62">Base62</a>
 * @author Leonard Woo
 */
public class Base62Digit {

  private static final int BASE_COUNT = 62;

  private static final char[] ALPHABET_BASE = new char[] {
      '0', '1', '2', '3', '4', '5', '6', '7', '8', '9',
      'A', 'B', 'C', 'D', 'E', 'F', 'G', 'H', 'I', 'J', 'K', 'L', 'M',
      'N', 'O', 'P', 'Q', 'R', 'S', 'T', 'U', 'V', 'W', 'X', 'Y', 'Z',
      'a', 'b', 'c', 'd', 'e', 'f', 'g', 'h', 'i', 'j', 'k', 'l', 'm',
      'n', 'o', 'p', 'q', 'r', 's', 't', 'u', 'v', 'w', 'x', 'y', 'z',
  };

  /**
   * encode number to Base62 string
   *
   * @param source number
   * @return Base62
   */
  public static String encode(long source) {
    StringBuilder sb = new StringBuilder();
    if (source == 0) {
      sb.append(ALPHABET_BASE[0]);
    }
    while (source > 0) {
      long mod = source % BASE_COUNT;
      sb.append(ALPHABET_BASE[(int) mod]);
      source = source / BASE_COUNT;
    }
    return sb.toString();
  }

  /**
   * decode Base62 string to number
   *
   * @param str Base62
   * @return number
   */
  public static long decode(String str) {
    final char[] items = str.toCharArray();
    long n = 0;
    for(int i = 0, pow = 0; i < items.length; i++, pow++) {
      // convert char to char array index
      int m = 0;
      if(NumberUtil.between(items[i], '0', '9')) {
        m = items[i] - '0';
      }
      if(NumberUtil.between(items[i], 'A', 'Z')) {
        m = items[i] - 'A' + 10;
      }
      if(NumberUtil.between(items[i], 'a', 'z')) {
        m = items[i] - 'a' + 36;
      }
      // char array index multiply 62^time
      n += m * MathUtil.power(BASE_COUNT, pow).longValue();
    }
    return n;
  }

}
