/*
 * Copyright 2022 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.seppiko.commons.utils;

import java.nio.ByteBuffer;
import java.nio.CharBuffer;
import java.nio.charset.CharacterCodingException;
import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;

/**
 * Character Util
 *
 * @author Leonard Woo
 */
public class CharUtil {

  /** NUL or {@code '\0'} */
  public static final Character NULL = 0x00;

  /** HT or {@code '\t'} */
  public static final Character HORIZONTAL_TABULATION = 0x09;

  /** LF or {@code '\n'} */
  public static final Character LINE_FEED = 0x0A;

  /** VT or {@code '\v'} */
  public static final Character VERTICAL_TABULATION = 0x0B;

  /** FF or {@code '\f'} */
  public static final Character FORM_FEED = 0x0C;

  /** CR or {@code '\r'} */
  public static final Character CARRIAGE_RETURN = 0x0D;

  /** CRLF or {@code "\r\n"} */
  public static final String CRLF = String.valueOf(new char[] {CARRIAGE_RETURN, LINE_FEED});

  /** {@code '!'} */
  public static final Character EXCLAMATION_MARK = 0x21;

  /** {@code '"'} */
  public static final Character QUOTATION_MARK = 0x22;

  /** {@code '#'} */
  public static final Character NUMBER_SIGN = 0x23;

  /** {@code '$'} */
  public static final Character DOLLAR_SIGN = 0x24;

  /** {@code '%'} */
  public static final Character PERCENT_SIGN = 0x25;

  /** {@code '&'} */
  public static final Character AMPERSAND = 0x26;

  /** {@code '\''} */
  public static final Character APOSTROPHE = 0x27;

  /** {@code '('} */
  public static final Character LEFT_PARENTHESIS = 0x28;

  /** {@code ')'} */
  public static final Character RIGHT_PARENTHESIS = 0x29;

  /** {@code '*'} */
  public static final Character ASTERISK = 0x2A;

  /** {@code '+'} */
  public static final Character PLUS = 0x2B;

  /** {@code ','} */
  public static final Character COMMA = 0x2C;

  /** {@code '-'} */
  public static final Character HYPHEN_MINUS = 0x2D;

  /** {@code '.'} */
  public static final Character FULL_STOP = 0x2E;

  /** {@code '/'} */
  public static final Character SOLIDUS = 0x2F;

  /** {@code ':'} */
  public static final Character COLON = 0x3A;

  /** {@code ';'} */
  public static final Character SEMICOLON = 0x3B;

  /** {@code '<'} */
  public static final Character LESS_THAN_SIGN = 0x3C;

  /** {@code '='} */
  public static final Character EQUALS_SIGN = 0x3D;

  /** {@code '>'} */
  public static final Character GREATER_THAN_SIGN = 0x3E;

  /** {@code '?'} */
  public static final Character QUESTION_MARK = 0x3F;

  /** {@code '@'} */
  public static final Character COMMERCIAL_AT = 0x40;

  /** {@code '['} */
  public static final Character LEFT_SQUARE_BRACKET = 0x5B;

  /** {@code '\'} */
  public static final Character REVERSE_SOLIDUS = 0x5C;

  /** {@code ']'} */
  public static final Character RIGHT_SQUARE_BRACKET = 0x5D;

  /** {@code '^'} */
  public static final Character CIRCUMFLEX_ACCENT = 0x5E;

  /** {@code '_'} */
  public static final Character LOW_LINE = 0x5F;

  /** {@code '`'} */
  public static final Character GRAVE_ACCENT = 0x60;

  /** {@code '&#123;' } */
  public static final Character LEFT_CURLY_BRACKET = 0x7B;

  /** {@code '|'} */
  public static final Character VERTICAL_LINE = 0x7C;

  /** {@code '&#125;'} */
  public static final Character RIGHT_CURLY_BRACKET = 0x7D;

  /** {@code '~'} */
  public static final Character TILDE = 0x7E;

  /** {@code '！'} */
  public static final Character FULLWIDTH_EXCLAMATION_MARK = 0xFF01;

  /** {@code '＂'} */
  public static final Character FULLWIDTH_QUOTATION_MARK = 0xFF02;

  /** {@code '＃'} */
  public static final Character FULLWIDTH_NUMBER_SIGN = 0xFF03;

  /** {@code '＄'} */
  public static final Character FULLWIDTH_DOLLAR_SIGN = 0xFF04;

  /** {@code '％'} */
  public static final Character FULLWIDTH_PERCENT_SIGN = 0xFF05;

  /** {@code '＆'} */
  public static final Character FULLWIDTH_AMPERSAND = 0xFF06;

  /** {@code '＇'} */
  public static final Character FULLWIDTH_APOSTROPHE = 0xFF07;

  /** {@code '（'} */
  public static final Character FULLWIDTH_LEFT_PARENTHESIS = 0xFF08;

  /** {@code '）'} */
  public static final Character FULLWIDTH_RIGHT_PARENTHESIS = 0xFF09;

  /** {@code '＊'} */
  public static final Character FULLWIDTH_ASTERISK = 0xFF0A;

  /** {@code '＋'} */
  public static final Character FULLWIDTH_PLUS = 0xFF0B;

  /** {@code '，'} */
  public static final Character FULLWIDTH_COMMA = 0xFF0C;

  /** {@code '－'} */
  public static final Character FULLWIDTH_HYPHEN_MINUS = 0xFF0D;

  /** {@code '．'} */
  public static final Character FULLWIDTH_FULL_STOP = 0xFF0E;

  /** {@code '／'} */
  public static final Character FULLWIDTH_SOLIDUS = 0xFF0F;

  /** {@code '：'} */
  public static final Character FULLWIDTH_COLON = 0xFF1A;

  /** {@code '；'} */
  public static final Character FULLWIDTH_SEMICOLON = 0xFF1B;

  /** {@code '＜'} */
  public static final Character FULLWIDTH_LESS_THAN_SIGN = 0xFF1C;

  /** {@code '＝'} */
  public static final Character FULLWIDTH_EQUALS_SIGN = 0xFF1D;

  /** {@code '＞'} */
  public static final Character FULLWIDTH_GREATER_THAN_SIGN = 0xFF1E;

  /** {@code '？'} */
  public static final Character FULLWIDTH_QUESTION_MARK = 0xFF1F;

  /** {@code '＠'} */
  public static final Character FULLWIDTH_COMMERCIAL_AT = 0xFF20;

  /** {@code '［'} */
  public static final Character FULLWIDTH_LEFT_SQUARE_BRACKET = 0xFF3B;

  /** {@code '＼'} */
  public static final Character FULLWIDTH_REVERSE_SOLIDUS = 0xFF3C;

  /** {@code '］'} */
  public static final Character FULLWIDTH_RIGHT_SQUARE_BRACKET = 0xFF3D;

  /** {@code '＾'} */
  public static final Character FULLWIDTH_CIRCUMFLEX_ACCENT = 0xFF3E;

  /** {@code '＿'} */
  public static final Character FULLWIDTH_LOW_LINE = 0xFF3F;

  /** {@code '｀'} */
  public static final Character FULLWIDTH_GRAVE_ACCENT = 0xFF40;

  /** {@code '｛'} */
  public static final Character FULLWIDTH_LEFT_CURLY_BRACKET = 0xFF5B;

  /** {@code '｜'} */
  public static final Character FULLWIDTH_VERTICAL_LINE = 0xFF5C;

  /** {@code '｝'} */
  public static final Character FULLWIDTH_RIGHT_CURLY_BRACKET = 0xFF5D;

  /** {@code '～'} */
  public static final Character FULLWIDTH_TILDE = 0xFF5E;

  /**
   * Charset decode
   *
   * @param charset charset, e.g. {@code StandardCharsets.UTF_8}
   * @param data byte array
   * @return character buffer object
   * @see Charset
   * @see StandardCharsets
   */
  public static CharBuffer charsetDecode(Charset charset, byte[] data) {
    try {
      return charset.newDecoder().decode(ByteBuffer.wrap(data));
    } catch (RuntimeException | CharacterCodingException ignored) {
    }
    return null;
  }

  /**
   * Charset encode
   *
   * @param charset charset, e.g. {@code StandardCharsets.UTF_8}
   * @param data character buffer
   * @return byte array
   * @see Charset
   * @see StandardCharsets
   */
  public static byte[] charsetEncode(Charset charset, CharBuffer data) {
    try {
      return charset.newEncoder().encode(data).array();
    } catch (RuntimeException | CharacterCodingException ignored) {
    }
    return null;
  }

  /**
   * Determines if the specified character is a digit.
   *
   * <p>Some Unicode character ranges that contain digits:
   *
   * <ul>
   *   <li>{@code '\u005Cu0030'} through {@code '\u005Cu0039'}, ISO-LATIN-1 digits ({@code '0'}
   *       through {@code '9'})
   *   <li>{@code '\u005CuFF10'} through {@code '\u005CuFF19'}, Fullwidth digits ({@code '０'}
   *       through {@code '９'})
   * </ul>
   *
   * @param ch the character to be tested.
   * @return true, if the character is a digit; false, otherwise.
   */
  public static boolean isDigit(char ch) {
    return NumberUtil.between(ch, 0x30, 0x39) || NumberUtil.between(ch, 0xFF10, 0xFF19);
  }

  /**
   * Determines if the specified character is a full stop.
   *
   * <ul>
   *   <li>{@code '\u005Cu002E'}, ISO-LATIN-1 full stop ({@code '.'}
   *   <li>{@code '\u005CuFF0E'}, Fullwidth digits ({@code '．'}
   * </ul>
   *
   * @param ch the character to be tested.
   * @return true, if the character is a digit; false, otherwise.
   */
  public static boolean isFullStop(char ch) {
    return ch == FULL_STOP || ch == FULLWIDTH_FULL_STOP;
  }
}
