/*
 * Copyright 2022 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.seppiko.commons.utils.codec;

import java.nio.charset.StandardCharsets;
import java.util.Base64;
import org.seppiko.commons.utils.BytesUtil;

/**
 * Base64 Encoder / Decoder base on Java 8+
 *
 * @see Base64
 * @author Leonard Woo
 */
public class Base64Util {

  /**
   * Get BASE 64 code
   *
   * @param data data source
   * @return Base64 Bytes
   */
  public static byte[] encode(byte[] data) {
    return Base64.getEncoder().encode(data);
  }

  /**
   * Get BASE 64 code to String
   *
   * @param data data source
   * @return Base64
   */
  public static String encodeToString(byte[] data) {
    return BytesUtil.decodeBytes(StandardCharsets.ISO_8859_1, encode(data)).toString();
  }

  /**
   * BASE 64 decode
   *
   * @param data BASE 64 source
   * @return BASE 64 decode
   */
  public static byte[] decode(byte[] data) {
    return Base64.getDecoder().decode(data);
  }

  /**
   * BASE 64 decode from String
   *
   * @param data BASE 64 source
   * @return BASE 64 decode
   */
  public static byte[] decode(String data) {
    return decode( StandardCharsets.ISO_8859_1.encode(data).array() );
  }

  /**
   * BASE 64 decode to string
   *
   * @param data string data
   * @return string
   */
  public static String decodeToString(String data) {
    return new String( decode(data) );
  }

  /**
   * BASE 64 decode to string
   *
   * @param data byte array data
   * @return string
   */
  public static String decodeToString(byte[] data) {
    return new String( decode(data) );
  }

  /**
   * Get Base64 code with URL
   * @param data data source
   * @return Base64
   */
  public static byte[] encodeUrl (byte[] data) {
    return Base64.getUrlEncoder().encode(data);
  }

  /**
   * Get BASE 64 code to String with URL
   *
   * @param data data source
   * @return Base64
   */
  public static String encodeToStringUrl (byte[] data) {
    return BytesUtil.decodeBytes(StandardCharsets.ISO_8859_1, encodeUrl(data)).toString();
  }

  /**
   * BASE 64 decode with URL
   *
   * @param data BASE 64 source
   * @return BASE 64 decode
   */
  public static byte[] decodeUrl (byte[] data) {
    return Base64.getUrlDecoder().decode(data);
  }

  /**
   * BASE 64 decode from String with URL
   *
   * @param data BASE 64 source
   * @return BASE 64 decode
   */
  public static byte[] decodeUrl (String data) {
    return decodeUrl( StandardCharsets.ISO_8859_1.encode(data).array() );
  }

}
