/*
 * Copyright 2022 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.seppiko.commons.utils.http;

import java.io.IOException;
import java.io.InputStream;
import java.net.InetSocketAddress;
import java.net.ProxySelector;
import java.net.URI;
import java.net.URISyntaxException;
import java.net.http.HttpClient;
import java.net.http.HttpRequest;
import java.net.http.HttpResponse;
import java.net.http.HttpResponse.BodyHandlers;
import java.util.concurrent.CompletableFuture;
import org.seppiko.commons.utils.ObjectUtil;

/**
 * Http Async Response Util with java.net.http
 *
 * <p>HttpRequest see {@link HttpClientUtil}
 *
 * @author Leonard Woo
 */
public class AsyncHttpClientUtil {

  /**
   * Get Async String Response
   *
   * @param req HttpRequest object
   * @param proxy Not null is enable proxy
   * @return Exception is NULL
   * @throws HttpResponseException if async response exception
   * @throws IllegalArgumentException if the request argument is not a request that could have been
   *     validly built as specified
   * @throws HttpRuntimeException http response field exception
   */
  public static HttpResponse<String> getAsyncResponseString(
      HttpRequest req, InetSocketAddress proxy)
      throws HttpRuntimeException, IllegalArgumentException, HttpResponseException {
    try {
      return httpResponse(getAsyncResponseRaw(req, BodyHandlers.ofString(), proxy));
    } catch (IOException e) {
      throw new HttpResponseException(e);
    }
  }

  /**
   * Get Async byte array Response
   *
   * @param req HttpRequest object
   * @param proxy Not null is enable proxy
   * @return Exception is NULL
   * @throws HttpResponseException get async response exception
   * @throws IllegalArgumentException if the request argument is not a request that could have been
   *     validly built as specified
   * @throws HttpRuntimeException http response field exception
   */
  public static HttpResponse<byte[]> getAsyncResponseByteArray(
      HttpRequest req, InetSocketAddress proxy)
      throws HttpRuntimeException, IllegalArgumentException, HttpResponseException {
    try {
      return httpResponse(getAsyncResponseRaw(req, BodyHandlers.ofByteArray(), proxy));
    } catch (IOException e) {
      throw new HttpResponseException(e);
    }
  }

  /**
   * Get Async InputStream Response
   *
   * @param req HttpRequest object
   * @param proxy Not null is enable proxy
   * @return Exception is NULL
   * @throws HttpResponseException get async response exception
   * @throws IllegalArgumentException if the request argument is not a request that could have been
   *     validly built as specified
   * @throws HttpRuntimeException http response field exception
   */
  public static HttpResponse<InputStream> getAsyncResponseInputStream(
      HttpRequest req, InetSocketAddress proxy)
      throws HttpRuntimeException, IllegalArgumentException, HttpResponseException {
    try {
      return httpResponse(getAsyncResponseRaw(req, BodyHandlers.ofInputStream(), proxy));
    } catch (IOException e) {
      throw new HttpResponseException(e);
    }
  }

  /**
   * send async http request
   *
   * @param url URL
   * @param method request http method
   * @param timeout request timeout
   * @param headers request headers, if none is null
   * @param requestBody request body, is none is null
   * @param responseType response type, default is byte array
   * @param proxy request http proxy, if had not is null
   * @param <R> request type (String, InputStream and byte[])
   * @param <T> response type (String, InputStream and byte[])
   * @return http response
   * @throws URISyntaxException URL is an illegal address
   * @throws HttpClientException get response exception
   * @throws HttpRuntimeException http request failed or the operation is interrupted.
   */
  @SuppressWarnings("unchecked")
  public static <R, T> HttpResponse<T> sendAsyncRequest(String url, HttpMethod method, int timeout,
      HttpHeaders headers, R requestBody, Class<T> responseType, InetSocketAddress proxy)
      throws URISyntaxException, HttpClientException, HttpRuntimeException {
    if (headers == null) {
      headers = HttpHeaders.newHeaders();
    }
    try {
      HttpRequest request = HttpClientUtil.getRequestRaw(new URI(url), method, timeout,
          headers.getHeaderList(), HttpClientUtil.getBodyPublisher(requestBody));
      return (HttpResponse<T>) httpResponse(
          getAsyncResponseRaw(request, HttpClientUtil.getBodyHandler(responseType), proxy));
    } catch (IOException | UnsupportedOperationException ex) {
      throw new HttpClientException(ex.getMessage());
    } catch (IllegalArgumentException | IllegalStateException ex) {
      throw new HttpRuntimeException(ex);
    }
  }

  private static <T> HttpResponse<T> httpResponse(CompletableFuture<HttpResponse<T>> resp)
      throws HttpRuntimeException {
    return new HttpResponseImpl<>(resp);
  }

  private static <T> CompletableFuture<HttpResponse<T>> getAsyncResponseRaw(
      HttpRequest req, HttpResponse.BodyHandler<T> responseBodyHandler, InetSocketAddress proxy)
      throws IllegalArgumentException, IOException {
    HttpClient.Builder builder = HttpClient.newBuilder();
    if (ObjectUtil.notNull(proxy)) {
      builder = builder.proxy(ProxySelector.of(proxy));
    }
    return builder.build().sendAsync(req, responseBodyHandler);
  }
}
