/*
 * Copyright 2022 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.seppiko.commons.utils;

import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.function.Function;
import java.util.function.Supplier;
import java.util.stream.Collectors;

/**
 * Collections Util
 *
 * @author Leonard Woo
 */
public class CollectionUtil {

  /**
   * Populates a Map using the supplied {@code Function} to transform the elements into keys, using
   * the unaltered element as the value in the {@code Map}. If the key is duplicated, only the new
   * key-value is retained.
   *
   * @param elements the {@code Collection} containing the input values for the map.
   * @param keyFunction the {@code Function} used to transform the element into a key value.
   * @return the {@code Map} to populate.
   * @param <K> the key type.
   * @param <V> the value type.
   */
  public static <K, V> Map<K, V> populateMap(
      final Collection<? extends V> elements, Function<? super V, ? extends K> keyFunction) {
    return populateMap(elements, keyFunction, Function.identity());
  }

  /**
   * Populates a Map using the supplied {@code Function}s to transform the elements into keys and
   * values. If the key is duplicated, only the new key-value is retained.
   *
   * @param elements the {@code Collection} containing the input values for the map.
   * @param keyFunction the {@code Function} used to transform the element into a key value.
   * @param valueFunction the {@code Function} used to transform the element into a value.
   * @return the {@code Map} to populate.
   * @param <K> the key type.
   * @param <V> the value type.
   * @param <E> the type of object contained in the {@link Collection}.
   */
  public static <K, V, E> Map<K, V> populateMap(
      final Collection<? extends E> elements,
      Function<? super E, ? extends K> keyFunction,
      Function<? super E, ? extends V> valueFunction) {
    return elements.stream()
        .collect(Collectors.toMap(keyFunction, valueFunction, (key1, key2) -> key2));
  }

  /**
   * Populates a Map using the supplied {@code Function}s to transform the elements into keys and
   * values.
   *
   * @param elements the {@code Collection} containing the input values for the map.
   * @param keyFunction the {@code Function} used to transform the element into a key value.
   * @param valueFunction the {@code Function} used to transform the element into a value.
   * @return the {@code Map} to populate.
   * @param <K> the key type.
   * @param <V> the value type.
   * @param <E> the type of object contained in the {@link Collection}.
   * @throws IllegalStateException Duplicate keys
   */
  public static <K, V, E> Map<K, V> populateMapWithoutFilter(
      final Collection<? extends E> elements,
      Function<? super E, ? extends K> keyFunction,
      Function<? super E, ? extends V> valueFunction)
      throws IllegalStateException {
    return elements.stream().collect(Collectors.toMap(keyFunction, valueFunction));
  }

  /**
   * Populates a List using the supplied {@code Function}s to transform the fields into values.
   *
   * @param elements Collection object
   * @param valueFunction Object field function
   * @return Field list
   * @param <T> the type of object contained in the {@link List}
   * @param <E> the type of object contained in the {@link Collection}
   */
  public static <T, E> List<T> populateList(
      final Collection<? extends E> elements, Function<? super E, ? extends T> valueFunction) {
    return elements.stream().map(valueFunction).collect(Collectors.toList());
  }

  /**
   * Returns a clone that accumulates the input elements into a new {@code Collection}, in encounter
   * order.
   *
   * @param collection source collection object e.g. {@code List} {@code Set} or {@code Map}
   * @param supplier a supplier providing a new empty {@code Collection} into which the results will
   *     be inserted
   * @return a {@code Collector} which collects all the input elements into a {@code Collection}, in
   *     encounter order
   * @param <T> the type of the input elements
   * @param <C> the type of the resulting {@code Collection}
   */
  public static <T, C extends Collection<T>> Collection<T> clone(
      Collection<? extends T> collection, Supplier<C> supplier) {
    return collection.stream().collect(Collectors.toCollection(supplier));
  }

}
