/*
 * Copyright 2022 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.seppiko.commons.utils.crypto.spec;

import java.math.BigInteger;
import java.security.spec.AlgorithmParameterSpec;
import java.security.spec.ECField;
import java.security.spec.ECParameterSpec;
import java.security.spec.ECPoint;
import java.security.spec.EllipticCurve;
import javax.crypto.spec.DHParameterSpec;
import javax.crypto.spec.GCMParameterSpec;
import javax.crypto.spec.IvParameterSpec;
import javax.crypto.spec.PBEParameterSpec;

/**
 * ParamterSpac implement util
 *
 * @author Leonard Woo
 */
public class ParamterSpacUtil {

  /**
   * GCMParameterSpec object
   *
   * @see GCMParameterSpec
   * @param tLen GCM authentication tag length
   * @param src the IV source buffer
   * @return GCMParameterSpec object
   * @throws NullPointerException src is null
   * @throws IllegalArgumentException if tLen is negative, src is null, len or offset is negative,
   * or the sum of offset and len is greater than the length of the src byte array.
   */
  public static GCMParameterSpec getGCM(int tLen, byte[] src)
      throws NullPointerException, IllegalArgumentException  {
    if (src == null) {
      throw new NullPointerException("src missing");
    }
    return getGCM(tLen, src, 0, src.length);
  }

  /**
   * GCMParameterSpec object
   *
   * @see GCMParameterSpec
   * @param tLen GCM authentication tag length
   * @param src the IV source buffer
   * @param off the offset in src where the IV starts
   * @param len the number of IV bytes
   * @return GCMParameterSpec object
   * @throws IllegalArgumentException if tLen is negative, src is null, len or offset is negative,
   * or the sum of offset and len is greater than the length of the src byte array.
   */
  public static GCMParameterSpec getGCM(int tLen, byte[] src, int off, int len)
      throws IllegalArgumentException {
    return new GCMParameterSpec(tLen, src, off, len);
  }

  /**
   * IvParameterSpec object
   *
   * @see IvParameterSpec
   * @param iv the buffer with the IV
   * @return IvParameterSpec object
   * @throws NullPointerException if iv is null
   * @throws IllegalArgumentException if iv is null or (iv.length - offset &lt; len)
   * @throws ArrayIndexOutOfBoundsException is thrown if offset or len index bytes outside the iv.
   */
  public static IvParameterSpec getIV(byte[] iv)
      throws NullPointerException {
    if (iv == null) {
      throw new NullPointerException("IV missing");
    }
    return getIV(iv, 0 , iv.length);
  }

  /**
   * IvParameterSpec object
   *
   * @see IvParameterSpec
   * @param iv the buffer with the IV
   * @param off the offset in iv where the IV starts
   * @param len the number of IV bytes
   * @return IvParameterSpec object
   * @throws IllegalArgumentException if iv is null or (iv.length - offset &lt; len)
   * @throws ArrayIndexOutOfBoundsException is thrown if offset or len index bytes outside the iv.
   */
  public static IvParameterSpec getIV(byte[] iv, int off, int len) {
    return new IvParameterSpec(iv, off, len);
  }

  /**
   * PBEParameterSpec object
   *
   * @see PBEParameterSpec
   * @param salt the salt. The contents of salt are copied to protect against subsequent modification.
   * @param iterationCount the iteration count.
   * @return PBEParameterSpec object
   * @throws NullPointerException if salt is null
   */
  public static PBEParameterSpec getPBE(byte[] salt, int iterationCount)
      throws NullPointerException {
    return getPBE(salt, iterationCount, null);
  }

  /**
   * PBEParameterSpec object
   *
   * @see PBEParameterSpec
   * @param salt the salt. The contents of salt are copied to protect against subsequent modification.
   * @param iterationCount the iteration count.
   * @param paramSpec the cipher algorithm parameter specification
   * @return PBEParameterSpec object
   * @throws NullPointerException if salt is null
   */
  public static PBEParameterSpec getPBE(byte[] salt, int iterationCount,
      AlgorithmParameterSpec paramSpec) throws NullPointerException {
    if (salt == null) {
      throw new NullPointerException("IV missing");
    }
    return new PBEParameterSpec(salt, iterationCount, paramSpec);
  }

  /**
   * DHParameterSpec object
   *
   * @see DHParameterSpec
   * @param p the prime modulus
   * @param g the base generator
   * @return DHParameterSpec object
   */
  public static DHParameterSpec getDH(BigInteger p, BigInteger g) {
    return getDH(p, g, 0);
  }

  /**
   * DHParameterSpec object
   *
   * @see DHParameterSpec
   * @param p the prime modulus
   * @param g the base generator
   * @param l the size in bits of the random exponent (private value)
   * @return DHParameterSpec object
   */
  public static DHParameterSpec getDH(BigInteger p, BigInteger g, int l) {
    return new DHParameterSpec(p, g, l);
  }

  /**
   *
   * @param field the finite field that this elliptic curve is over.
   * @param fristCoefficient the first coefficient of this elliptic curve.
   * @param secondfficient the second coefficient of this elliptic curve.
   * @param affineX the affine x-coordinate.
   * @param affineY the affine y-coordinate.
   * @param generator the order of the generator.
   * @param cofactor the cofactor.
   * @return ECParameterSpec object
   * @throws NullPointerException if same parameter is null
   * @throws IllegalArgumentException if {@code generator} or {@code cofactor} is not positive or
   *         {@code affineX} or {@code affineY} is not null and not in {@code field}.
   */
  public static ECParameterSpec getEC(ECField field, BigInteger fristCoefficient, BigInteger secondfficient,
      BigInteger affineX, BigInteger affineY, BigInteger generator, int cofactor)
      throws NullPointerException, IllegalArgumentException {
    EllipticCurve curve = new EllipticCurve(field, fristCoefficient, secondfficient);
    ECPoint point = new ECPoint(affineX, affineY);
    return new ECParameterSpec(curve, point, generator, cofactor);
  }

}
