/*
 * Copyright 2023 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.seppiko.commons.utils;

/**
 * Assert
 *
 * @author Leonard Woo
 */
public class Assert {

  /**
   * if not true throw message
   *
   * @param expression boolean expression.
   * @param message message.
   * @throws IllegalArgumentException if expression is false throw this.
   */
  public static void isTrue(boolean expression, String message) {
    if (!expression) {
      throw new IllegalArgumentException(message);
    }
  }

  /**
   * if true throw message
   *
   * @param expression boolean expression.
   * @param message message.
   * @throws IllegalArgumentException if expression is true throw this.
   */
  public static void isFalse(boolean expression, String message) {
    if (expression) {
      throw new IllegalArgumentException(message);
    }
  }

  /**
   * if string text length is 0 throw message
   *
   * @param text string text.
   * @param message message.
   * @throws NullPointerException if text have not any string throw this.
   */
  public static void hasLength(String text, String message) {
    if (StringUtil.isNullOrEmpty(text)) {
      throw new NullPointerException(message);
    }
  }

  /**
   * if string text has not any char without whitespace throw message
   *
   * @param text string text.
   * @param message message.
   * @throws NullPointerException if text have not any char throw this.
   */
  public static void hasText(String text, String message) {
    if (StringUtil.nonText(text)) {
      throw new NullPointerException(message);
    }
  }

  /**
   * Throw exception with RuntimeException.
   *
   * <pre><code>
   *  try {
   *    throw new Throwable();
   *  } catch (Throwable t) {
   *    throw Assert.sneakyThrow(t);
   *  }
   * </code></pre>
   *
   * @param t any exception.
   * @return RuntimeException.
   * @throws NullPointerException if t is null throw this.
   */
  public static RuntimeException sneakyThrow(Throwable t) {
    if (t == null) throw new NullPointerException();
    return Assert.sneakyThrow0(t);
  }

  @SuppressWarnings("unchecked")
  private static <T extends Throwable> T sneakyThrow0(Throwable t) throws T {
    throw (T) t;
  }
}
