/*
 * Copyright 2023 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.seppiko.commons.utils;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.util.Arrays;
import java.util.List;
import java.util.Objects;
import java.util.function.Supplier;

/**
 * Object Util
 *
 * @see Objects
 * @see java.util.Optional
 * @author Leonard Woo
 */
public class ObjectUtil {

  private ObjectUtil() {}

  /**
   * Returns {@code true} if the provided reference is array,
   * otherwise returns {@code false}.
   *
   * @param obj a reference to be checked against.
   * @param <T> reference type.
   * @return {@code true} if the provided reference is array, otherwise {@code false}.
   */
  public static <T> boolean isArray(T obj) {
    return Objects.nonNull(obj) && obj.getClass().isArray();
  }

  /**
   * Returns {@code true} if the array is {@code null},
   * otherwise returns {@code false}.
   *
   * @param array a reference to be checked against.
   * @param <T> reference type.
   * @return {@code true} if the provided array is {@code null}, otherwise {@code false}.
   */
  public static <T> boolean isEmpty(T[] array) {
    return (isNull(array) || array.length == 0);
  }

  /**
   * Returns {@code true} if the provided reference is {@code null},
   * otherwise returns {@code false}.
   *
   * @param obj a reference to be checked against {@code null}.
   * @param <T> reference type.
   * @return {@code true} if the provided reference is {@code null}, otherwise {@code false}.
   */
  public static <T> boolean isNull(T[] obj) {
    return null == obj;
  }

  /**
   * Returns {@code true} if the provided reference is non-{@code null},
   * otherwise returns {@code false}.
   *
   * @param obj a reference to be checked against {@code null}.
   * @param <T> reference type.
   * @return {@code true} if the provided reference is non-{@code null}, otherwise {@code false}.
   */
  public static <T> boolean nonNull(T[] obj) {
    return null != obj;
  }

  /**
   * Creates and returns a deep copy.
   *
   * @param obj an object.
   * @return a deep copy of the object.
   * @throws IOException copy failed.
   * @throws ReflectiveOperationException object cannot be found.
   * @throws SecurityException if the stream header is incorrect.
   * @throws NullPointerException object is null.
   */
  public static Object deepCopy(Object obj)
      throws IOException, ReflectiveOperationException, SecurityException, NullPointerException {
    Objects.requireNonNull(obj);
    try (ByteArrayOutputStream baos = new ByteArrayOutputStream();
        ObjectOutputStream oos = new ObjectOutputStream(baos)) {
      oos.writeObject(obj);
      oos.flush();
      try (ObjectInputStream ois = new ObjectInputStream(new ByteArrayInputStream(baos.toByteArray()))) {
        return ois.readObject();
      }
    }
  }

  /**
   * Get Supplier object with not null array.
   *
   * @param suppliers supplier array.
   * @return object array.
   * @throws NullPointerException supplier array is null.
   */
  public static Object[] getSuppliers(Supplier<?>[] suppliers) throws NullPointerException {
    Objects.requireNonNull(suppliers);
    List<Supplier<?>> list = Arrays.stream(suppliers).filter(Objects::nonNull).toList();
    Object[] objs = new Object[list.size()];
    for (int i = 0; i < list.size(); i++) {
      objs[i] = list.get(i).get();
    }
    return objs;
  }
}
