/*
 * Copyright 2023 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.seppiko.commons.utils;

import java.nio.CharBuffer;
import java.nio.charset.Charset;
import java.util.Objects;
import java.util.StringJoiner;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.regex.PatternSyntaxException;

/**
 * String Util
 *
 * @author Leonard Woo
 */
public class StringUtil {

  private StringUtil() {}

  /**
   * Require a CharSequence with default value.
   *
   * @param value CharSequence instance.
   * @param defaultValue default CharSequence instance.
   * @return CharSequence without null.
   */
  public static CharSequence requireNonBlankElse(CharSequence value, CharSequence defaultValue) {
    return isNullOrEmpty(value) ? defaultValue : value;
  }

  /**
   * Test CharSequence is empty or contains only whitespace codepoints
   *
   * @param value CharSequence instance.
   * @return true is yes.
   */
  public static boolean isNullOrEmpty(CharSequence value) {
    return Objects.isNull(value) || value.length() == 0;
  }

  /**
   * Test CharSequence has any char
   *
   * @param str CharSequence instance.
   * @return true is yes.
   */
  public static boolean hasLength(CharSequence str) {
    return (Objects.nonNull(str) && str.length() > 0);
  }

  /**
   * Test CharSequence has any char without non-blank character
   *
   * @param str CharSequence instance.
   * @return true is not null and without whitespace, false is otherwise.
   */
  public static boolean hasText(CharSequence str) {
    return (hasLength(str) && containsText(str));
  }

  /**
   * Test CharSequence is not null and empty or has non-blank character
   *
   * @param str CharSequence instance.
   * @return true is not null and empty, false is otherwise.
   */
  public static boolean nonText(CharSequence str) {
    return Objects.nonNull(str) && (str.length() == 0 || !containsText(str));
  }

  private static boolean containsText(CharSequence str) {
    int strLen = str.length();
    for (int i = 0; i < strLen; i++) {
      char ch = str.charAt(i);
      if (!Character.isWhitespace(ch) || !Character.isISOControl(ch)) {
        return true;
      }
    }
    return false;
  }

  /**
   * Test CharSequence is numeric
   *
   * @param input numeric CharSequence.
   * @return true is numeric.
   */
  public static boolean isNumeric(CharSequence input) {
    return Boolean.TRUE.equals(matches("^[-－]?[0-9０-９]+([\\.．]{1}[0-9０-９]+)?$", input));
  }

  /**
   * Test CharSequence is integer
   *
   * @param input numeric CharSequence.
   * @return true is integer.
   */
  public static boolean isDigit(CharSequence input) {
    return Boolean.TRUE.equals(matches("^[-－]?[0-9０-９]+$", input));
  }

  /**
   * Test CharSequence is decimal
   *
   * @param input numeric CharSequence.
   * @return true is decimal.
   */
  public static boolean isDecimal(CharSequence input) {
    return Boolean.TRUE.equals(matches("^[-－]?[0-9０-９]+[\\.．]{1}[0-9０-９]+$", input));
  }

  /**
   * Test CharSequence is punctuation
   *
   * @param input punctuation CharSequence.
   * @return true is punctuation.
   */
  public static boolean isPunctuation(CharSequence input) {
    return Boolean.TRUE.equals(matches("^[\\p{Punct}！＂＃＄％＆＇（）＊＋，－．／：；＜＝＞？＠［＼］＾＿｀｛｜｝]+$", input));
  }

  /**
   * Compiles the given regular expression and attempts to match the given input against it.
   *
   * @param regex The expression to be compiled.
   * @param input The character sequence to be matched.
   * @return whether the regular expression matches on the input, when the expression's syntax is
   *     invalid is {@code null}.
   */
  public static Boolean matches(String regex, CharSequence input) {
    try {
      return Objects.requireNonNull(getMatcher(regex, input)).matches();
    } catch (NullPointerException ignored) {
    }
    return null;
  }

  /**
   * Compiles the given regular expression the given input against it.
   *
   * @param regex The expression to be compiled.
   * @param input The character sequence to be matched.
   * @return {@link Matcher} instance, if the expression's syntax is invalid return null.
   */
  public static Matcher getMatcher(String regex, CharSequence input) {
    try {
      Pattern pattern = Pattern.compile(regex);
      return pattern.matcher(input);
    } catch (PatternSyntaxException ignored) {
    }
    return null;
  }

  /**
   * Convert string data from old encoding to new encoding
   *
   * @param data CharSequence data.
   * @param oldEncoding old encoding.
   * @param newEncoding new encoding.
   * @return new encoding CharSequence.
   * @throws NullPointerException old encode or new decode exception.
   */
  public static String transcoding(CharSequence data, Charset oldEncoding, Charset newEncoding)
      throws NullPointerException {
    byte[] oldData = CharUtil.charsetEncode(oldEncoding, CharBuffer.wrap(data));
    if (oldData == Environment.EMPTY_BYTE_ARRAY) {
      throw new NullPointerException("old encode exception");
    }
    CharBuffer cb = CharUtil.charsetDecode(newEncoding, oldData);
    if (cb.capacity() == 0) {
      throw new NullPointerException("new decode exception");
    }
    return cb.toString();
  }

  /**
   * Return fixed length string object
   *
   * @param str string object.
   * @param length count length.
   * @param preChar pre-padded character.
   * @return fixed length string object.
   */
  public static String fixedLength(String str, int length, char preChar) {
    StringBuilder sb = new StringBuilder();
    if (str.length() < length) {
      int preLength = length - str.length();
      sb.append(String.valueOf(preChar).repeat(preLength));
      sb.append(str);
    } else {
      sb.append(str.substring(str.length() - length));
    }
    return sb.toString();
  }

  /**
   * Capitalize the first letter
   *
   * @param input origin string.
   * @return new string.
   */
  public static String toFirstUpperCase(String input) {
    String s = String.valueOf(Character.toUpperCase(input.charAt(0)));
    if (input.length() > 1) {
      return s + input.subSequence(1, input.length());
    }
    return s;
  }

  /**
   * Replace CharSequence between start and end
   *
   * @param data origin data.
   * @param start replace start index.
   * @param end replace end index.
   * @param replacement replace data.
   * @return new string.
   */
  public static String replaceBetween(
      CharSequence data, int start, int end, CharSequence replacement) {
    StringBuilder sb = new StringBuilder();
    sb.append(data, 0, start);
    sb.append(replacement);
    sb.append(data, end, data.length());
    return sb.toString();
  }

  /**
   * convert joiner string with delimiter
   *
   * @param delimiter the sequence of characters to be used between each array element.
   * @param strs string array.
   * @return the string representation.
   */
  public static String convertJoinerString(String delimiter, String... strs) {
    StringJoiner sj = new StringJoiner(delimiter);
    for (String str : strs) {
      if (hasLength(str)) {
        sj.add(str);
      }
    }
    return sj.toString();
  }

  /**
   * Convert char array to String with separate
   *
   * @param src Raw data.
   * @param splitNum Separation interval.
   * @param split Separator.
   * @return encoded string.
   */
  public static String convertToString(char[] src, int splitNum, String split) {
    StringBuilder sb = new StringBuilder();
    for (int i = 0; i < src.length; i = i + splitNum) {
      for (int j = 0; j < splitNum; j++) {
        sb.append(src[i + j]);
      }
      sb.append(split);
    }
    String dst = sb.toString();
    return dst.substring(0, dst.length() - split.length());
  }

  /**
   * Delete string Separator and to char array.
   *
   * @param src string.
   * @param split separator.
   * @return char array.
   * @throws NullPointerException when data or separator is null.
   */
  public static char[] convertToCharArray(String src, String split) throws NullPointerException {
    Objects.requireNonNull(src, "src must not be null.");
    Objects.requireNonNull(split, "split must not be null.");
    Matcher m = getMatcher(split, src);
    Objects.requireNonNull(m, "split matcher failed");
    if (m.find()) {
      return m.replaceAll("").toCharArray();
    }
    return src.toCharArray();
  }

  private static final Integer BETWEEN =
      (CharUtil.FULLWIDTH_EXCLAMATION_MARK - CharUtil.EXCLAMATION_MARK);

  /**
   * Convert half-width string ({@code '\u005Cu0021'} through {@code '\u005Cu007E'}) to
   * full-width string ({@code '\u005CuFF01'} through {@code '\u005CuFF5E'})
   *
   * @param src Half-width string.
   * @return Full-width string.
   */
  public static String toFullWidth(CharSequence src) {
    CharBuffer cb = CharBuffer.allocate(src.length());
    for (int i = 0; i < cb.length(); i++) {
      char c = src.charAt(i);
      if (NumberUtil.between(c, CharUtil.EXCLAMATION_MARK, CharUtil.TILDE)) {
        cb.append((char) (c + BETWEEN));
      } else {
        cb.append(c);
      }
    }
    return cb.toString();
  }

  /**
   * Convert full-width string ({@code '\u005CuFF01'} through {@code '\u005CuFF5E'}) to
   * half-width string ({@code '\u005Cu0021'} through {@code '\u005Cu007E'})
   *
   * @param src Full-width string.
   * @return Half-width string.
   */
  public static String toHalfWidth(CharSequence src) {
    CharBuffer cb = CharBuffer.allocate(src.length());
    for (int i = 0; i < cb.length(); i++) {
      char c = src.charAt(i);
      if (NumberUtil.between(c, CharUtil.FULLWIDTH_EXCLAMATION_MARK, CharUtil.FULLWIDTH_TILDE)) {
        cb.append((char) (c - BETWEEN));
      } else {
        cb.append(c);
      }
    }
    return cb.toString();
  }

  /**
   * Unicode decoding When not found return origin String
   *
   * @param src unicode
   * @return String
   */
  public static String unicodeDecode(String src) {
    Matcher m = getMatcher("(\\\\u(\\p{XDigit}{4}))", src);
    if (m == null) {
      return src;
    }
    while (m.find()) {
      String group = m.group(2);
      char ch = (char) Integer.parseInt(group, 16);
      String group1 = m.group(1);
      src = src.replace(group1, ch + "");
    }
    return src;
  }

  /**
   * Unicode encoding
   *
   * @param src String
   * @return unicode
   */
  public static String unicodeEncode(String src) {
    if (isNullOrEmpty(src)) {
      return src;
    }
    StringBuilder sb = new StringBuilder();
    for (int i = 0; i < src.length(); i++) {
      char ch = src.charAt(i);
      if (NumberUtil.between(ch, CharUtil.SPACE, CharUtil.TILDE)) {
        sb.append(ch);
      } else {
        sb.append("\\u");
        String hex = Integer.toHexString(ch & 0xFFFF).toUpperCase();
        sb.append(fixedLength(hex, 4, CharUtil.DIGIT_ONE));
      }
    }
    return sb.toString();
  }
}
