/*
 * Copyright 2023 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.seppiko.commons.utils.codec;

import java.math.BigInteger;

/**
 * Base36 digit.
 * A simple base36 encode / decode.
 * Commonly used by URL redirection systems like TinyURL or SnipURL as
 * compact alphanumeric identifiers.
 *
 * @see <a href="https://en.wikipedia.org/wiki/Base36">Base36</a>
 * @author Leonard Woo
 */
public class Base36Digit {

  private Base36Digit() {}

  /**
   * Encode number to Base36 string
   *
   * @param source Number.
   * @return Base36 String.
   */
  public static String encode(final long source) {
    return encode(source, false);
  }

  /**
   * Encode number to Base36 string
   *
   * @param source Number.
   * @param toLowercase true is return {@code [0-9a-z]}, false is {@code [0-9A-Z]}.
   * @return Base36 String.
   */
  public static String encode(final long source, final boolean toLowercase) {
    String base62 = BigInteger.valueOf(source).toString(36);
    return toLowercase? base62: base62.toUpperCase();
  }

  /**
   * Decode Base36 string to number
   *
   * @param str Base36 String.
   * @return Number.
   */
  public static long decode(final String str) {
    return new BigInteger(str, 36).longValue();
  }
}
