/*
 * Copyright 2023 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.seppiko.commons.utils.codec;

import org.seppiko.commons.utils.CharUtil;
import org.seppiko.commons.utils.Environment;
import org.seppiko.commons.utils.NumberUtil;
import org.seppiko.commons.utils.StringUtil;

/**
 * Hex Encoder / Decoder Util.
 *
 * @author Leonard Woo
 */
public class HexUtil {

  private HexUtil() {}

  private static final int DIGIT_LENGTH = Environment.DIGIT_LENGTH;

  /** Hexadecimal length */
  public static final int HEXADECIMAL_LENGTH = 16;

  /** Hexadecimal character array with lower letters */
  public static final char[] HEXADECIMAL = new char[HEXADECIMAL_LENGTH];

  /** Hexadecimal character array with upper letters */
  public static final char[] HEXADECIMAL_UPPER = new char[HEXADECIMAL_LENGTH];

  static {
    System.arraycopy(CharUtil.DIGIT, 0, HEXADECIMAL, 0, DIGIT_LENGTH);
    System.arraycopy(CharUtil.LOWERCASE, 0, HEXADECIMAL, DIGIT_LENGTH, (HEXADECIMAL_LENGTH - DIGIT_LENGTH));

    System.arraycopy(CharUtil.DIGIT, 0, HEXADECIMAL_UPPER, 0, DIGIT_LENGTH);
    System.arraycopy(CharUtil.UPPERCASE, 0, HEXADECIMAL_UPPER, DIGIT_LENGTH, (HEXADECIMAL_LENGTH - DIGIT_LENGTH));
  }

  /**
   * Returns the numeric value of the hexadecimal character {@code ch}
   *
   * @param ch the character to be converted
   * @return the numeric value represented by the hexadecimal character
   * @see Character#digit(char, int)
   */
  public static int hexDigit(char ch) {
    int i = -1;
    if (NumberUtil.between(ch, CharUtil.DIGIT_ZERO, CharUtil.DIGIT_NINE)) {
      i = ch - CharUtil.DIGIT_ZERO;
    } else if (NumberUtil.between(ch, CharUtil.UPPERCASE_A, HEXADECIMAL_UPPER[HEXADECIMAL_LENGTH - 1])) {
      i = (char) (ch - CharUtil.UPPERCASE_A) + DIGIT_LENGTH;
    } else if (NumberUtil.between(ch, CharUtil.LOWERCASE_A, HEXADECIMAL[HEXADECIMAL_LENGTH - 1])) {
      i = (char) (ch - CharUtil.LOWERCASE_A) + DIGIT_LENGTH;
    }
    return i;
  }

  /**
   * convert byte array to string object
   *
   * @param data byte array data
   * @return string object
   */
  public static String encodeString(byte[] data) {
    return encodeString(data, " ");
  }

  /**
   * convert byte array to string object
   *
   * @param data byte array data
   * @param split string object split
   * @return string object
   */
  public static String encodeString(byte[] data, String split) {
    return StringUtil.convertToString(encode(data), 2, split);
  }

  /**
   * convert byte array to hex char array
   *
   * @param data byte array data
   * @return hex char array
   */
  public static char[] encode(byte[] data) {
    return encode(data, true);
  }

  /**
   * convert byte array to hex char array
   *
   * @param data byte array data
   * @param toLowerCase true is lower case, false is upper case
   * @return hex char array
   */
  public static char[] encode(byte[] data, boolean toLowerCase) {
    return encode(data, toLowerCase? HEXADECIMAL: HEXADECIMAL_UPPER);
  }

  private static char[] encode(byte[] data, char[] hexChar) {
    char[] hex = new char[data.length * 2];
    for (int i = 0, j = 0; i < data.length; i++, j = j + 2) {
      byte b = data[i];
      hex[j] = hexChar[(0xF0 & b) >> 0x04];
      hex[j + 1] = hexChar[0x0F & b];
    }
    return hex;
  }

  /**
   * convert hex string with whitespace split to byte array
   *
   * @param data hex string
   * @return byte array
   * @throws IllegalArgumentException data include invalid character
   * @throws NullPointerException when data or separator is null
   */
  public static byte[] decodeString(String data)
      throws IllegalArgumentException, NullPointerException {
    return decode(data, " ");
  }

  /**
   * convert hex string with split to byte array
   *
   * @param data hex string
   * @param split split
   * @return byte array
   * @throws IllegalArgumentException data include invalid character
   * @throws NullPointerException when data or separator is null
   */
  public static byte[] decode(String data, String split)
      throws IllegalArgumentException, NullPointerException {
    return decode(StringUtil.convertToCharArray(data, split));
  }

  /**
   * convert hex char array to byte array
   *
   * @param data hex char array
   * @return byte array
   * @throws IllegalArgumentException data include invalid character
   * @throws NullPointerException data is null or empty
   */
  public static byte[] decode(char[] data) throws IllegalArgumentException, NullPointerException {
    if (null == data || data.length == 0) {
      throw new NullPointerException();
    }
    byte[] hex = new byte[data.length / 2];
    for (int i = 0, j = 0; i < hex.length; i++, j = j + 2) {
      int f = toDigit(data[j]) << 0x04;
      f = f | toDigit(data[j + 1]);
      hex[i] = (byte) (f & 0xFF);
    }
    return hex;
  }

  private static int toDigit(char hexChar) throws IllegalArgumentException {
    int digit = hexDigit(hexChar);
    if (digit < 0) {
      throw new IllegalArgumentException("Invalid Hexadecimal Character: " + hexChar);
    }
    return digit;
  }
}
