/*
 * Copyright 2023 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.seppiko.commons.utils.crypto;

import java.io.Serial;
import java.io.Serializable;
import java.security.NoSuchAlgorithmException;
import java.util.Objects;
import org.seppiko.commons.utils.StringUtil;

/**
 * Algorithm Name/Mode/Padding util
 *
 * @author Leonard Woo
 */
public class NMPUtil implements Serializable {

  @Serial private static final long serialVersionUID = -215141540814890802L;

  /** algorithm name */
  private final String name;

  /** algorithm mode */
  private final String mode;

  /** algorithm padding */
  private final String padding;

  /**
   * Parser algorithm String
   *
   * @param algorithm Name[/Mode[/Padding]]
   * @throws NoSuchAlgorithmException Algorithm String format is wrong
   */
  public NMPUtil(String algorithm) throws NoSuchAlgorithmException {
    if (algorithm.indexOf('/') < 0) {
      this.name = algorithm;
      this.mode = null;
      this.padding = null;
    } else {
      String[] strs = algorithm.split("\\/");
      if (strs.length == 2) {
        this.name = strs[0];
        this.mode = strs[1];
        this.padding = null;
      } else if (strs.length == 3) {
        this.name = strs[0];
        this.mode = strs[1];
        this.padding = strs[2];
      } else {
        throw new NoSuchAlgorithmException("Algorithm string parser failed.");
      }
    }
  }

  /**
   * Init NMP
   *
   * @param name name
   * @param mode mode
   * @param padding padding
   */
  public NMPUtil(String name, String mode, String padding) {
    this.name = name;
    this.mode = mode;
    this.padding = padding;
  }

  /**
   * get name
   *
   * @return name
   */
  public String getName() {
    return name;
  }

  /**
   * get mode
   *
   * @return mode
   */
  public String getMode() {
    return mode;
  }

  /**
   * get padding
   *
   * @return padding
   */
  public String getPadding() {
    return padding;
  }

  /**
   * Check nmp object
   *
   * @param o nmp object
   * @return true, if object is NMPUtil and equals
   */
  @Override
  public boolean equals(Object o) {
    if (this == o) {
      return true;
    }
    if (o == null || getClass() != o.getClass()) {
      return false;
    }

    NMPUtil that = (NMPUtil) o;
    return Objects.equals(name, that.name)
        && Objects.equals(mode, that.mode)
        && Objects.equals(padding, that.padding);
  }

  /** NMPUtil object hashcode */
  @Override
  public int hashCode() {
    return Objects.hash(name, mode, padding);
  }

  /** Re-splicing algorithm name */
  @Override
  public String toString() {
    return StringUtil.convertJoinerString("/", name, mode, padding);
  }
}
