/*
 * Copyright 2023 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.seppiko.commons.utils.http;

import java.io.IOException;
import java.io.InputStream;
import java.net.InetSocketAddress;
import java.net.ProxySelector;
import java.net.URI;
import java.net.URISyntaxException;
import java.net.http.HttpClient;
import java.net.http.HttpRequest;
import java.net.http.HttpResponse;
import java.net.http.HttpResponse.BodyHandlers;
import java.time.Duration;
import java.util.Objects;

/**
 * Http Client Util with java.net.http
 *
 * @author Leonard Woo
 */
public class HttpClientUtil {

  private HttpClientUtil() {}

  /**
   * Get Request object with String
   *
   * @param url URL
   * @param method see {@link HttpMethod}
   * @param timeout Timeout
   * @param headers Headers
   * @param requestBody Request body string
   * @return HttpRequest object
   * @throws URISyntaxException URL is an illegal address
   * @throws IllegalArgumentException something not supported or undefined
   * @throws NullPointerException something is null
   * @throws HttpRuntimeException Http request process exception
   */
  public static HttpRequest getRequest(
      String url, HttpMethod method, int timeout, HttpHeaders headers, String requestBody)
      throws URISyntaxException, IllegalArgumentException, NullPointerException,
          HttpRuntimeException {

    if (Objects.isNull(headers)) {
      headers = HttpHeaders.newHeaders();
    }

    HttpRequest.BodyPublisher requestBodyPublisher =
        Objects.nonNull(requestBody)
            ? HttpRequest.BodyPublishers.ofString(requestBody)
            : HttpRequest.BodyPublishers.noBody();

    return getRequestRaw(
        new URI(url), method, timeout, headers.getHeaderList(), requestBodyPublisher);
  }

  /**
   * Get Request object with inputstream
   *
   * @param url URL
   * @param method see {@link HttpMethod}
   * @param timeout Timeout
   * @param headers Headers
   * @param requestBody Request body inputstream
   * @return HttpRequest object
   * @throws URISyntaxException URL is an illegal address
   * @throws IllegalArgumentException something not supported or undefined
   * @throws NullPointerException something is null
   * @throws HttpRuntimeException Http request process exception
   */
  public static HttpRequest getRequest(
      String url, HttpMethod method, int timeout, HttpHeaders headers, InputStream requestBody)
      throws URISyntaxException, IllegalArgumentException, NullPointerException,
          HttpRuntimeException {

    if (Objects.isNull(headers)) {
      headers = HttpHeaders.newHeaders();
    }

    HttpRequest.BodyPublisher requestBodyPublisher =
        Objects.nonNull(requestBody)
            ? HttpRequest.BodyPublishers.ofInputStream(() -> requestBody)
            : HttpRequest.BodyPublishers.noBody();

    return getRequestRaw(
        new URI(url), method, timeout, headers.getHeaderList(), requestBodyPublisher);
  }

  /**
   * Get Request object with byte array
   *
   * @param url URL
   * @param method see {@link HttpMethod}
   * @param timeout Timeout
   * @param headers Headers
   * @param requestBody Request body byte array
   * @return HttpRequest object
   * @throws URISyntaxException URL is an illegal address
   * @throws IllegalArgumentException something not supported or undefined
   * @throws NullPointerException something is null
   * @throws HttpRuntimeException Http request process exception
   */
  public static HttpRequest getRequest(
      String url, HttpMethod method, int timeout, HttpHeaders headers, byte[] requestBody)
      throws URISyntaxException, IllegalArgumentException, NullPointerException,
          HttpRuntimeException {

    if (Objects.isNull(headers)) {
      headers = HttpHeaders.newHeaders();
    }

    HttpRequest.BodyPublisher requestBodyPublisher =
        Objects.nonNull(requestBody)
            ? HttpRequest.BodyPublishers.ofByteArray(requestBody)
            : HttpRequest.BodyPublishers.noBody();

    return getRequestRaw(
        new URI(url), method, timeout, headers.getHeaderList(), requestBodyPublisher);
  }

  /**
   * Get Response with String body
   *
   * @param req HttpRequest object
   * @param proxy Http proxy, Not null is enabled proxy
   * @return HttpResponse object, if NULL has failed
   * @throws HttpClientException get response exception
   * @throws HttpInterruptedException if the operation is interrupted
   */
  public static HttpResponse<String> getResponseString(HttpRequest req, InetSocketAddress proxy)
      throws HttpClientException, HttpInterruptedException {
    return getResponseRaw(req, BodyHandlers.ofString(), proxy);
  }

  /**
   * Get Response with byte array body
   *
   * @param req HttpRequest object
   * @param proxy Http proxy, Not null is enabled proxy
   * @return HttpResponse object, if NULL has failed
   * @throws HttpClientException get response exception
   * @throws HttpInterruptedException if the operation is interrupted
   */
  public static HttpResponse<byte[]> getResponseByteArray(HttpRequest req, InetSocketAddress proxy)
      throws HttpClientException, HttpInterruptedException {
    return getResponseRaw(req, BodyHandlers.ofByteArray(), proxy);
  }

  /**
   * Get Response with inputstream body
   *
   * @param req HttpRequest object
   * @param proxy Http proxy, Not null is enabled proxy
   * @return HttpResponse object, if NULL has failed
   * @throws HttpClientException get response exception
   * @throws HttpInterruptedException if the operation is interrupted
   */
  public static HttpResponse<InputStream> getResponseInputStream(
      HttpRequest req, InetSocketAddress proxy)
      throws HttpClientException, HttpInterruptedException {
    return getResponseRaw(req, BodyHandlers.ofInputStream(), proxy);
  }

  /**
   * send http request
   *
   * @param url URL
   * @param method request http method
   * @param timeout request timeout
   * @param headers request headers, if none is null
   * @param requestBody request body, is none is null
   * @param responseType response type, default is byte array
   * @param proxy request http proxy, if had not is null
   * @param <R> request type (String, InputStream and byte[])
   * @param <T> response type (String, InputStream and byte[])
   * @return http response
   * @throws URISyntaxException URL is an illegal address
   * @throws HttpClientException get response exception
   * @throws HttpRuntimeException http request failed or the operation is interrupted
   * @throws HttpTimeoutException http request timeout
   */
  @SuppressWarnings("unchecked")
  public static <R, T> HttpResponse<T> sendRequest(
      String url,
      HttpMethod method,
      int timeout,
      HttpHeaders headers,
      R requestBody,
      Class<T> responseType,
      InetSocketAddress proxy)
      throws URISyntaxException, HttpClientException, HttpRuntimeException, HttpTimeoutException {

    if (headers == null) {
      headers = HttpHeaders.newHeaders();
    }
    try {
      HttpRequest request =
          getRequestRaw(
              new URI(url),
              method,
              timeout,
              headers.getHeaderList(),
              getBodyPublisher(requestBody));
      return (HttpResponse<T>) getResponseRaw(request, getBodyHandler(responseType), proxy);
    } catch (UnsupportedOperationException ex) {
      throw new HttpClientException(ex.getMessage());
    } catch (IllegalArgumentException | IllegalStateException ex) {
      throw new HttpRuntimeException(ex);
    }
  }

  /**
   * Get HTTP request raw
   *
   * @param uri request uri
   * @param method request method
   * @param timeout request timeout
   * @param headers request headers
   * @param requestBody request body
   * @return HttpRequest object
   * @throws IllegalArgumentException URI method or headers is not support or valid
   * @throws IllegalStateException if a URI has not been set
   * @throws HttpTimeoutException timeout is not valid
   */
  protected static HttpRequest getRequestRaw(
      URI uri,
      HttpMethod method,
      int timeout,
      String[] headers,
      HttpRequest.BodyPublisher requestBody)
      throws IllegalArgumentException, IllegalStateException, HttpTimeoutException {
    HttpRequest.Builder builder =
        HttpRequest.newBuilder().uri(uri).method(method.name(), requestBody);
    if (headers.length > 0) {
      builder = builder.headers(headers);
    }
    try {
      builder = builder.timeout(Duration.ofSeconds(timeout));
    } catch (IllegalArgumentException ex) {
      throw new HttpTimeoutException("HTTP request timeout.");
    }
    return builder.build();
  }

  /**
   * Get HTTP response raw
   *
   * @param req HTTP request
   * @param responseBodyHandler Response body content type
   * @param proxy HTTP Proxy, Not null is enabled proxy
   * @return Http response
   * @param <T> Response content type
   * @throws HttpInterruptedException If the response is interrupted
   * @throws HttpResponseException If the response has I/O error or not valid
   * @throws SecurityException If a security manager has been installed and it denies access to the
   *     URL in the given request, or proxy if one is configured
   */
  private static <T> HttpResponse<T> getResponseRaw(
      HttpRequest req, HttpResponse.BodyHandler<T> responseBodyHandler, InetSocketAddress proxy)
      throws HttpInterruptedException, HttpResponseException, SecurityException {
    try {
      HttpClient.Builder builder = HttpClient.newBuilder();
      if (Objects.nonNull(proxy)) {
        builder = builder.proxy(ProxySelector.of(proxy));
      }
      return builder.build().send(req, responseBodyHandler);
    } catch (InterruptedException ex) {
      throw new HttpInterruptedException(ex);
    } catch (IOException | IllegalArgumentException ex) {
      throw new HttpResponseException(ex);
    }
  }

  /**
   * Request body checker
   *
   * @param requestBody request body
   * @return BodyPublisher object
   * @param <T> body type
   * @throws UnsupportedOperationException not found request body class type
   */
  protected static <T> HttpRequest.BodyPublisher getBodyPublisher(T requestBody)
      throws UnsupportedOperationException {
    if (Objects.isNull(requestBody)) {
      return HttpRequest.BodyPublishers.noBody();
    }
    Class<?> type = requestBody.getClass();
    if (type.isAssignableFrom(CharSequence.class)) {
      return HttpRequest.BodyPublishers.ofString(requestBody.toString());
    }
    if (type.isAssignableFrom(InputStream.class)) {
      return HttpRequest.BodyPublishers.ofInputStream(() -> (InputStream) requestBody);
    }
    if (requestBody instanceof byte[]) {
      return HttpRequest.BodyPublishers.ofByteArray((byte[]) requestBody);
    }
    throw new UnsupportedOperationException("Can NOT found supported type");
  }

  /**
   * Response body checker
   *
   * @param type response type
   * @return BodyHandler type
   * @param <T> class type
   */
  protected static <T> HttpResponse.BodyHandler<?> getBodyHandler(Class<T> type) {
    if (type == null || type.isAssignableFrom(String.class)) {
      return BodyHandlers.ofString();
    }
    if (type.isAssignableFrom(InputStream.class)) {
      return BodyHandlers.ofInputStream();
    }
    return BodyHandlers.ofByteArray();
  }
}
