/*
 * Copyright 2023 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.seppiko.commons.utils.http;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.ConcurrentModificationException;
import java.util.LinkedHashMap;
import org.seppiko.commons.utils.StringUtil;

/**
 * Simple Http Headers implement for {@link java.net.http.HttpRequest}
 *
 * @author Leonard Woo
 */
public class HttpHeaders {

  private static final char HTTP_HEADER_VALUE_SEPARATOR = ';';
  private final LinkedHashMap<String, ArrayList<String>> headers = new LinkedHashMap<>();
  private final String[] EMPTY_STRING_ARRAY = new String[0];

  private HttpHeaders() {}

  /**
   * create http headers
   *
   * @return new HttpHeaders object
   */
  public static HttpHeaders newHeaders() {
    return new HttpHeaders();
  }

  /**
   * Set header
   *
   * @param name header name
   * @param values header values
   * @return this
   */
  public HttpHeaders setHeaders(String name, String... values) {
    headers.put(name, new ArrayList<>(Arrays.asList(values)));
    return this;
  }

  /**
   * Add headers
   *
   * @param name header name
   * @param values header values
   * @return this
   * @throws IllegalArgumentException copy old values failed
   */
  public HttpHeaders addHeaders(String name, String... values) throws IllegalArgumentException {
    if (headers.containsKey(name)) {
      ArrayList<String> valueOld = headers.get(name);
      ArrayList<String> valueNew = new ArrayList<>(Arrays.asList(values));
      for (String value : valueOld) {
        if (!valueNew.contains(value)) {
          try {
            valueNew.add(value);
          } catch (UnsupportedOperationException
              | ClassCastException
              | IllegalArgumentException
              | NullPointerException ex) {
            throw new IllegalArgumentException(ex);
          }
        }
      }
      headers.put(name, valueNew);
    } else {
      setHeaders(name, values);
    }
    return this;
  }

  /**
   * Get {@link java.net.http.HttpRequest} headers
   *
   * @return Headers
   * @throws IllegalCallerException Object init args exception
   * @throws IllegalArgumentException Args exception
   */
  public String[] getHeaderList() throws IllegalCallerException, IllegalArgumentException {
    return (headers.isEmpty()) ? EMPTY_STRING_ARRAY : headerHandler(headers);
  }

  private String[] headerHandler(LinkedHashMap<String, ArrayList<String>> headerMap)
      throws IllegalCallerException, IllegalArgumentException {
    ArrayList<String> headerList = new ArrayList<>();
    try {
      headerMap.forEach(
          (key, value) -> {
            headerList.add(key);
            headerList.add(valuesHandler(value));
          });

      return headerList.toArray(String[]::new);
    } catch (ConcurrentModificationException | ClassCastException | ArrayStoreException ex) {
      throw new IllegalCallerException(ex);
    } catch (UnsupportedOperationException | IllegalArgumentException ex) {
      throw new IllegalArgumentException(ex);
    } catch (NullPointerException ignored) {
    }
    return EMPTY_STRING_ARRAY;
  }

  private String valuesHandler(ArrayList<String> valueList) {
    return StringUtil.convertJoinerString(
        HTTP_HEADER_VALUE_SEPARATOR + " ", valueList.toArray(String[]::new));
  }
}
