/*
 * Copyright 2023 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.seppiko.commons.utils;

import java.util.Calendar;
import java.util.Objects;

/**
 * Calendar Utility
 *
 * @see Calendar
 * @author Leonard Woo
 */
public class CalendarUtil {

  private CalendarUtil() {}

  /**
   * The {@link Calendar#DAY_OF_WEEK} field indicating.
   */
  public enum DayOfWeekShort {

    /**
     * Value of the {@link Calendar#DAY_OF_WEEK} field indicating
     * Sunday.
     */
    SUN(Calendar.SUNDAY, "Sun"),

    /**
     * Value of the {@link Calendar#DAY_OF_WEEK} field indicating
     * Monday.
     */
    MON(Calendar.MONDAY, "Mon"),

    /**
     * Value of the {@link Calendar#DAY_OF_WEEK} field indicating
     * Tuesday.
     */
    TUE(Calendar.TUESDAY, "Tue"),

    /**
     * Value of the {@link Calendar#DAY_OF_WEEK} field indicating
     * Wednesday.
     */
    WED(Calendar.WEDNESDAY, "Wed"),

    /**
     * Value of the {@link Calendar#DAY_OF_WEEK} field indicating
     * Thursday.
     */
    THU(Calendar.THURSDAY, "Thu"),

    /**
     * Value of the {@link Calendar#DAY_OF_WEEK} field indicating
     * Friday.
     */
    FRI(Calendar.FRIDAY, "Fri"),

    /**
     * Value of the {@link Calendar#DAY_OF_WEEK} field indicating
     * Saturday.
     */
    SAT(Calendar.SATURDAY, "Sat"),
    ;

    private final int value;
    private final String formatName;
    DayOfWeekShort(int value, String formatName) {
      this.value = value;
      this.formatName = formatName;
    }

    private static final DayOfWeekShort[] VALUES;
    static {
      VALUES = values();
    }

    /**
     * Get day of week code
     *
     * @return code
     */
    public int getDayOfWeekCode() {
      return value;
    }

    /**
     * Get day of week name
     *
     * @return name
     */
    public String getFormatName() {
      return formatName;
    }

    /**
     * Return an {@link DayOfWeekShort} object for the given value.
     *
     * @param name the method value as a String.
     * @return the corresponding {@code HttpMethod}, or {@code null} if not found
     * @throws NullPointerException method name is null.
     */
    public static DayOfWeekShort findByName(String name) throws NullPointerException {
      Objects.requireNonNull(name);
      final String finalName = name.strip().toUpperCase();
      for (DayOfWeekShort value : VALUES) {
        if (value.name().equals(finalName)) {
          return value;
        }
      }
      return null;
    }
  }

  /**
   * The {@link Calendar#MONTH} field indicating.
   */
  public enum MonthShort {
    /**
     * Value of the {@link Calendar#MONTH} field indicating the
     * first month of the year in the Gregorian and Julian calendars.
     */
    JAN (Calendar.JANUARY, "Jan"),

    /**
     * Value of the {@link Calendar#MONTH} field indicating the
     * second month of the year in the Gregorian and Julian calendars.
     */
    FEB(Calendar.FEBRUARY, "Feb"),

    /**
     * Value of the {@link Calendar#MONTH} field indicating the
     * third month of the year in the Gregorian and Julian calendars.
     */
    MAR(Calendar.MARCH, "Mar"),

    /**
     * Value of the {@link Calendar#MONTH} field indicating the
     * fourth month of the year in the Gregorian and Julian calendars.
     */
    APR(Calendar.APRIL, "Apr"),

    /**
     * Value of the {@link Calendar#MONTH} field indicating the
     * fifth month of the year in the Gregorian and Julian calendars.
     */
    MAY(Calendar.MAY, "May"),

    /**
     * Value of the {@link Calendar#MONTH} field indicating the
     * sixth month of the year in the Gregorian and Julian calendars.
     */
    JUN(Calendar.JUNE, "Jun"),

    /**
     * Value of the {@link Calendar#MONTH} field indicating the
     * seventh month of the year in the Gregorian and Julian calendars.
     */
    JUL(Calendar.JULY, "Jul"),

    /**
     * Value of the {@link Calendar#MONTH} field indicating the
     * eighth month of the year in the Gregorian and Julian calendars.
     */
    AUG(Calendar.AUGUST, "Aug"),

    /**
     * Value of the {@link Calendar#MONTH} field indicating the
     * ninth month of the year in the Gregorian and Julian calendars.
     */
    SEP(Calendar.SEPTEMBER, "Sep"),

    /**
     * Value of the {@link Calendar#MONTH} field indicating the
     * tenth month of the year in the Gregorian and Julian calendars.
     */
    OCT(Calendar.OCTOBER, "Oct"),

    /**
     * Value of the {@link Calendar#MONTH} field indicating the
     * eleventh month of the year in the Gregorian and Julian calendars.
     */
    NOV(Calendar.NOVEMBER, "Nov"),

    /**
     * Value of the {@link Calendar#MONTH} field indicating the
     * twelfth month of the year in the Gregorian and Julian calendars.
     */
    DEC(Calendar.DECEMBER, "Dec"),
    ;

    private final int value;
    private final String formatName;
    MonthShort(int value, String formatName) {
      this.value = value;
      this.formatName = formatName;
    }

    private static final MonthShort[] VALUES;
    static {
      VALUES = values();
    }

    /**
     * Get month code
     *
     * @return code
     */
    public int getMonth() {
      return value + 1;
    }

    /**
     * Get month name
     *
     * @return name
     */
    public String getFormatName() {
      return formatName;
    }

    /**
     * Return an {@link MonthShort} object for the given value.
     *
     * @param name the method value as a String.
     * @return the corresponding {@code HttpMethod}, or {@code null} if not found
     * @throws NullPointerException method name is null.
     */
    public static MonthShort findByName(String name) throws NullPointerException {
      Objects.requireNonNull(name);
      final String finalName = name.strip().toUpperCase();
      for (MonthShort value : VALUES) {
        if (value.name().equals(finalName)) {
          return value;
        }
      }
      return null;
    }
  }
}
