/*
 * Copyright 2023 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.seppiko.commons.utils;

import java.io.IOException;
import java.util.Objects;
import java.util.Optional;

/**
 * Object Util
 *
 * @see Objects
 * @see Optional
 * @author Leonard Woo
 */
public class ObjectUtil {

  private ObjectUtil() {}

  /**
   * Get value or else
   *
   * Like this
   * <pre><code>
   * Optional.ofNullable(value).orElse(defaultValue);
   * </code></pre>
   *
   * @param value the possibly-null value to describe.
   * @param other the value to be returned, if value is null. May be null.
   * @return the value, if value not null, otherwise other.
   * @param <T> the type of value.
   */
  public static <T> T requireWithElse(T value, T other) {
    return Objects.nonNull(value)? value: other;
  }

  /**
   * Get {@link Optional} with ignore value nullable.
   *
   * @param value the possibly-null value to describe.
   * @return the {@link Optional}.
   * @param <T> the type of value.
   */
  public static <T> Optional<T> optionalOf(T value) {
    return Optional.ofNullable(value);
  }

  /**
   * Creates and returns a deep copy.
   *
   * @param obj an object.
   * @return a deep copy of the object.
   * @throws IOException if an I/O error occurs while writing stream header.
   * @throws ReflectiveOperationException object cannot be found.
   * @throws SecurityException if the stream header is incorrect.
   * @throws NullPointerException object is null.
   */
  public static Object deepCopy(Object obj)
      throws IOException, ReflectiveOperationException, SecurityException, NullPointerException {
    Objects.requireNonNull(obj);
    byte[] _tmp = SerializationUtil.serialize(obj);
    return SerializationUtil.deserialize(_tmp, Object.class);
  }
}
