/*
 * Copyright 2023 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.seppiko.commons.utils.codec;

import java.io.Serial;
import java.io.Serializable;
import org.seppiko.commons.utils.StringUtil;

/**
 * Base16 Encoder / Decoder Util.
 *
 * @see <a href="https://datatracker.ietf.org/doc/html/rfc4648#section-8">RFC4648</a>
 * @author Leonard Woo
 */
public class Base16 extends BaseNCodec implements Serializable {

  @Serial
  private static final long serialVersionUID = -3844227952413070095L;

  /** Base16 character array with lower letters */
  private static final char[] RFC4648_ENCODE_TABLE = {
      '0', '1', '2', '3', '4', '5', '6', '7', '8', '9',
      'a', 'b', 'c', 'd', 'e', 'f'
  };

  /** Base16 character array with upper letters */
  private static final char[] RFC4648_UPPER_ENCODE_TABLE = {
      '0', '1', '2', '3', '4', '5', '6', '7', '8', '9',
      'A', 'B', 'C', 'D', 'E', 'F'
  };

  /** Base16 characters are 4 bits in length. */
  private static final int BITS_PER_ENCODED_BYTE = 4;

  private static final byte[] HEXADECIMAL_DECODE_TABLE = {
  //  0   1   2   3   4   5   6   7   8   9   A   B   C   D   E   F
      -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, // 00-0f
      -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, // 10-1f
      -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, // 20-2f
       0,  1,  2,  3,  4,  5,  6,  7,  8,  9, -1, -1, -1, -1, -1, -1, // 30-3f 0-9
      -1, 10, 11, 12, 13, 14, 15,                                     // 40-46 A-F
                                  -1, -1, -1, -1, -1, -1, -1, -1, -1, // 47-4f
      -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, // 50-5f
      -1, 10, 11, 12, 13, 14, 15                                      // 60-66 a-f
  };

  /**
   * The most widely used Base16 alphabet is defined in RFC 4648.
   */
  public static final Base16 RFC4648 = new Base16(RFC4648_ENCODE_TABLE);

  /**
   * The most widely used Base16 alphabet is defined in RFC 4648.
   */
  public static final Base16 RFC4648_UPPER = new Base16(RFC4648_UPPER_ENCODE_TABLE);

  private final char[] encodeTable;

  private Base16(char[] encodeTable) {
    this.encodeTable = encodeTable;
  }

  /**
   * convert byte array to Base16.
   *
   * @param data byte array data.
   * @return hex char array.
   */
  @Override
  public String encode(final byte[] data) {
    return new String(encode0(data, encodeTable));
  }

  private char[] encode0(byte[] data, char[] hexChar) {
    char[] hex = new char[data.length * 2];
    for (int i = 0, j = 0; i < data.length; i++) {
      byte b = data[i];
      hex[j++] = hexChar[(int) ((b >> BITS_PER_ENCODED_BYTE) & MASK_4BITS)];
      hex[j++] = hexChar[(int) (b & MASK_4BITS)];
    }
    return hex;
  }

  /**
   * convert Base16 to byte array.
   *
   * @param str hex char array.
   * @return byte array.
   * @throws IllegalArgumentException data include invalid character.
   * @throws NullPointerException data is null or empty.
   */
  @Override
  public byte[] decode(final String str) throws IllegalArgumentException, NullPointerException {
    if (StringUtil.isNullOrEmpty(str)) {
      throw new NullPointerException();
    }
    char[] data = str.toCharArray();
    byte[] hex = new byte[data.length / 2];
    for (int i = 0, j = 0; i < hex.length; i++) {
      int f = HEXADECIMAL_DECODE_TABLE[data[j++]] << BITS_PER_ENCODED_BYTE;
      f |= HEXADECIMAL_DECODE_TABLE[data[j++]];
      hex[i] = (byte) f;
    }
    return hex;
  }
}
