/*
 * Copyright 2023 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.seppiko.commons.utils.codec;

import org.seppiko.commons.utils.CharUtil;
import org.seppiko.commons.utils.Environment;
import org.seppiko.commons.utils.MathUtil;
import org.seppiko.commons.utils.NumberUtil;

/**
 * Base62 digit.
 * A simple base62 encode / decode.
 * Similar to base64, typically used in URL shortness.
 *
 * @see <a href="https://en.wikipedia.org/wiki/Base62">Base62</a>
 * @author Leonard Woo
 */
public class Base62Digit {

  private static final int BASE_COUNT = Environment.DIGIT_ALPHABET_ALL_COUNT;

  private static final char[] ALPHABET_BASE = Environment.DIGIT_ALPHABET_ALL;

  private Base62Digit() {}

  /**
   * Encode number to Base62 string
   *
   * @param source Number.
   * @return Base62 String.
   */
  public static String encode(final long source) {
    StringBuilder sb = new StringBuilder();
    if (source == 0) {
      sb.append(ALPHABET_BASE[0]);
    }
    long l = source;
    while (l > 0) {
      long mod = l % BASE_COUNT;
      sb.append(ALPHABET_BASE[(int) mod]);
      l = l / BASE_COUNT;
    }
    return sb.toString();
  }

  /**
   * Decode Base62 string to number
   *
   * @param str Base62 String.
   * @return Number.
   */
  public static long decode(final String str) {
    final char[] items = str.toCharArray();
    long n = 0;
    for (int i = 0, pow = 0; i < items.length; i++, pow++) {
      // convert char to char array index
      int m = 0;
      if (NumberUtil.between(items[i], CharUtil.DIGIT_ZERO, CharUtil.DIGIT_NINE)) {
        m = items[i] - CharUtil.DIGIT_ZERO;
      }
      if (NumberUtil.between(items[i], CharUtil.UPPERCASE_A, CharUtil.UPPERCASE_Z)) {
        m = items[i] - CharUtil.UPPERCASE_A + Environment.DIGIT_LENGTH;
      }
      if (NumberUtil.between(items[i], CharUtil.LOWERCASE_A, CharUtil.LOWERCASE_Z)) {
        m = items[i] - CharUtil.LOWERCASE_A + (Environment.DIGIT_LENGTH + Environment.ALPHABET_LENGTH);
      }
      // char array index multiply 62^time
      n += m * MathUtil.pow(BASE_COUNT, pow).longValue();
    }
    return n;
  }
}
