/*
 * Copyright 2021 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.seppiko.glf.api;

import java.io.IOException;
import java.net.URL;
import java.util.ArrayList;
import java.util.Enumeration;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.ServiceLoader;
import java.util.Set;
import org.seppiko.glf.event.DefaultLogger;
import org.seppiko.glf.event.DefaultServiceProvider;
import org.seppiko.glf.event.nop.NOPServiceProvider;
import org.seppiko.glf.spi.GLFServiceProvider;
import org.seppiko.glf.utils.GLFReport;

/**
 * Logger factory is a virtual implementation class for get real implementation service of logger.
 *
 * @author Leonard Woo
 */
public final class LoggerFactory {

  static volatile GLFServiceProvider PROVIDER;

  private static final int UNINITIALIZED = 0;
  private static final int ONGOING_INITIALIZATION = 1;
  private static final int FAILED_INITIALIZATION = 2;
  private static final int SUCCESSFUL_INITIALIZATION = 3;
  private static final int NOP_PROVIDER_INITIALIZATION = 4;

  private static volatile int STATUS = UNINITIALIZED;

  private static final DefaultServiceProvider DEFAULT_PROVIDER = new DefaultServiceProvider();
  private static final NOPServiceProvider NOP_PROVIDER = new NOPServiceProvider();

  private static final String STATIC_LOGGER_BINDER = "org/seppiko/glf/impl/StaticLoggerBinder.class";

  public static Logger getLogger(String name) {
    return getLoggerFactory().getLogger(name);
  }

  public static Logger getLogger(Class<?> clazz) {
    return getLogger(clazz.getName());
  }

  public static ILoggerFactory getLoggerFactory() {
    return getServiceProvider().getILoggerFactory();
  }

  public static GLFServiceProvider getServiceProvider() {
    if (STATUS == UNINITIALIZED) {
      synchronized (LoggerFactory.class) {
        if (STATUS == UNINITIALIZED) {
          STATUS = ONGOING_INITIALIZATION;
          bind();
        }
      }
    }

    switch (STATUS) {
      case SUCCESSFUL_INITIALIZATION:
        return PROVIDER;
      case NOP_PROVIDER_INITIALIZATION:
        return NOP_PROVIDER;
      case FAILED_INITIALIZATION:
        throw new IllegalStateException("LoggerFactory in failed state. Original exception was thrown EARLIER.");
      case ONGOING_INITIALIZATION:
        return DEFAULT_PROVIDER;
      default:
        throw new IllegalStateException("Unreachable code");
    }
  }

  private static void bind() {
    try {
      List<GLFServiceProvider> providers = findGLFServiceProvider();
      if (providers.size() > 1) {
        GLFReport.err("Class path contains multiple GLF providers.");
        for (GLFServiceProvider provider : providers) {
          GLFReport.err("Found provider [" + provider + "]");
        }
      }
      if (!providers.isEmpty()) {
        PROVIDER = providers.get(0);
        PROVIDER.initialize();
        STATUS = SUCCESSFUL_INITIALIZATION;

        if (providers.size() > 1) {
          GLFReport.info("Actual provider is of type [" + providers.get(0) + "]");
        }
      } else {
        STATUS = NOP_PROVIDER_INITIALIZATION;
        GLFReport.err("No GLF service providers were found.");
        GLFReport.err("Defaulting to no-operation (NOP) logger implementation");

        Set<URL> staticLoggerBinderPathSet = findPossibleStaticLoggerBinderPathSet();
        reportIgnoredStaticLoggerBinders(staticLoggerBinderPathSet);
      }

      synchronized (DEFAULT_PROVIDER) {
        for (DefaultLogger defaultLogger : DEFAULT_PROVIDER.getLoggerFactory().getLoggers()) {
          Logger logger = getLogger(defaultLogger.getName());
          defaultLogger.setDelegate(logger);
        }
      }

      DEFAULT_PROVIDER.getLoggerFactory().clear();
    } catch (Exception e) {
      STATUS = FAILED_INITIALIZATION;
      GLFReport.err("Failed to instantiate GLF LoggerFactory", e);
      throw new IllegalStateException("Unexpected initialization failure", e);
    }
  }

  private static List<GLFServiceProvider> findGLFServiceProvider() {
    ServiceLoader<GLFServiceProvider> loader = ServiceLoader.load(GLFServiceProvider.class);
    List<GLFServiceProvider> providers = new ArrayList<>();
    loader.forEach(providers::add);
    return providers;
  }

  static Set<URL> findPossibleStaticLoggerBinderPathSet() {
    Set<URL> staticLoggerBinderPathSet = new LinkedHashSet<>();
    try {
      ClassLoader loggerFactoryClassLoader = LoggerFactory.class.getClassLoader();
      Enumeration<URL> paths;
      if (loggerFactoryClassLoader == null) {
        paths = ClassLoader.getSystemResources(STATIC_LOGGER_BINDER);
      } else {
        paths = loggerFactoryClassLoader.getResources(STATIC_LOGGER_BINDER);
      }
      while (paths.hasMoreElements()) {
        URL path = paths.nextElement();
        staticLoggerBinderPathSet.add(path);
      }
    } catch (IOException ioe) {
      GLFReport.err("Error getting resources from path", ioe);
    }
    return staticLoggerBinderPathSet;
  }

  private static void reportIgnoredStaticLoggerBinders(Set<URL> staticLoggerBinderPathSet) {
    if (staticLoggerBinderPathSet.isEmpty()) {
      return;
    }
    GLFReport.err("Class path contains GLF bindings targeting glf-api.");
    for (URL path : staticLoggerBinderPathSet) {
      GLFReport.err("Ignoring binding found at [" + path + "]");
    }
  }
}
