/*
 * Copyright 2021 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.seppiko.glf.log4j2;

import org.apache.logging.log4j.message.Message;
import org.apache.logging.log4j.message.ParameterizedMessage;
import org.apache.logging.log4j.message.SimpleMessage;
import org.apache.logging.log4j.spi.ExtendedLogger;
import org.apache.logging.log4j.Level;
import org.seppiko.glf.api.LocationLogger;
import org.seppiko.glf.api.Marker;
import org.seppiko.glf.impl.StaticMarkerBinder;

/**
 * Log4j 2.x Logger implementation convert
 *
 * @author Leonard Woo
 */
public class Log4j2Logger implements LocationLogger {

  public static final String FQCN = Log4j2Logger.class.getName();

  private transient ExtendedLogger logger;
  private final String name;

  public Log4j2Logger(final ExtendedLogger logger, final String name) {
    this.logger = logger;
    this.name = name;
  }

  @Override
  public String getName() {
    return name;
  }

  @Override
  public boolean isTraceEnabled() {
    return logger.isEnabled(Level.TRACE, null, null);
  }

  @Override
  public boolean isTraceEnabled(Marker marker) {
    return logger.isEnabled(Level.TRACE, getMarker(marker), null);
  }

  @Override
  public void trace(String message) {
    logger.logIfEnabled(FQCN, Level.TRACE, null, message);
  }

  @Override
  public void trace(String message, Object param) {
    logger.logIfEnabled(FQCN, Level.TRACE, null, message, param);
  }

  @Override
  public void trace(String message, Object... params) {
    logger.logIfEnabled(FQCN, Level.TRACE, null, message, params);
  }

  @Override
  public void trace(String message, Throwable cause) {
    logger.logIfEnabled(FQCN, Level.TRACE, null, message, cause);
  }

  @Override
  public void trace(String message, Object param, Throwable cause) {
    logger.logIfEnabled(FQCN, Level.TRACE, null, message, param, cause);
  }

  @Override
  public void trace(String message, Object[] params, Throwable cause) {
    logger.logIfEnabled(FQCN, Level.TRACE, null, message, params, cause);
  }

  @Override
  public void trace(Marker marker, String message) {
    logger.logIfEnabled(FQCN, Level.TRACE, getMarker(marker), message);
  }

  @Override
  public void trace(Marker marker, String message, Object param) {
    logger.logIfEnabled(FQCN, Level.TRACE, getMarker(marker), message, param);
  }

  @Override
  public void trace(Marker marker, String message, Object... params) {
    logger.logIfEnabled(FQCN, Level.TRACE, getMarker(marker), message, params);
  }

  @Override
  public void trace(Marker marker, String message, Throwable cause) {
    logger.logIfEnabled(FQCN, Level.TRACE, getMarker(marker), message, cause);
  }

  @Override
  public void trace(Marker marker, String message, Object param, Throwable cause) {
    logger.logIfEnabled(FQCN, Level.TRACE, getMarker(marker), message, param, cause);
  }

  @Override
  public void trace(Marker marker, String message, Object[] params, Throwable cause) {
    logger.logIfEnabled(FQCN, Level.TRACE, getMarker(marker), message, params, cause);
  }

  @Override
  public boolean isDebugEnabled() {
    return logger.isEnabled(Level.DEBUG, null, null);
  }

  @Override
  public boolean isDebugEnabled(Marker marker) {
    return logger.isEnabled(Level.DEBUG, getMarker(marker), null);
  }

  @Override
  public void debug(String message) {
    logger.logIfEnabled(FQCN, Level.DEBUG, null, message);
  }

  @Override
  public void debug(String message, Object param) {
    logger.logIfEnabled(FQCN, Level.DEBUG, null, message, param);
  }

  @Override
  public void debug(String message, Object... params) {
    logger.logIfEnabled(FQCN, Level.DEBUG, null, message, params);
  }

  @Override
  public void debug(String message, Throwable cause) {
    logger.logIfEnabled(FQCN, Level.DEBUG, null, message, cause);
  }

  @Override
  public void debug(String message, Object param, Throwable cause) {
    logger.logIfEnabled(FQCN, Level.DEBUG, null, message, param, cause);
  }

  @Override
  public void debug(String message, Object[] params, Throwable cause) {
    logger.logIfEnabled(FQCN, Level.DEBUG, null, message, params, cause);
  }

  @Override
  public void debug(Marker marker, String message) {
    logger.logIfEnabled(FQCN, Level.DEBUG, getMarker(marker), message);
  }

  @Override
  public void debug(Marker marker, String message, Object param) {
    logger.logIfEnabled(FQCN, Level.DEBUG,  getMarker(marker), message, param);
  }

  @Override
  public void debug(Marker marker, String message, Object... params) {
    logger.logIfEnabled(FQCN, Level.DEBUG,  getMarker(marker), message, params);
  }

  @Override
  public void debug(Marker marker, String message, Throwable cause) {
    logger.logIfEnabled(FQCN, Level.DEBUG,  getMarker(marker), message, cause);
  }

  @Override
  public void debug(Marker marker, String message, Object param, Throwable cause) {
    logger.logIfEnabled(FQCN, Level.DEBUG,  getMarker(marker), message, param, cause);
  }

  @Override
  public void debug(Marker marker, String message, Object[] params, Throwable cause) {
    logger.logIfEnabled(FQCN, Level.DEBUG,  getMarker(marker), message, params, cause);
  }

  @Override
  public boolean isInfoEnabled() {
    return logger.isEnabled(Level.INFO, null, null);
  }

  @Override
  public boolean isInfoEnabled(Marker marker) {
    return logger.isEnabled(Level.INFO, getMarker(marker), null);
  }

  @Override
  public void info(String message) {
    logger.logIfEnabled(FQCN, Level.INFO, null, message);
  }

  @Override
  public void info(String message, Object param) {
    logger.logIfEnabled(FQCN, Level.INFO, null, message, param);
  }

  @Override
  public void info(String message, Object... params) {
    logger.logIfEnabled(FQCN, Level.INFO, null, message, params);
  }

  @Override
  public void info(String message, Throwable cause) {
    logger.logIfEnabled(FQCN, Level.INFO, null, message, cause);
  }

  @Override
  public void info(String message, Object param, Throwable cause) {
    logger.logIfEnabled(FQCN, Level.INFO, null, message, param, cause);
  }

  @Override
  public void info(String message, Object[] params, Throwable cause) {
    logger.logIfEnabled(FQCN, Level.INFO, null, message, params, cause);
  }

  @Override
  public void info(Marker marker, String message) {
    logger.logIfEnabled(FQCN, Level.INFO, getMarker(marker), message);
  }

  @Override
  public void info(Marker marker, String message, Object param) {
    logger.logIfEnabled(FQCN, Level.INFO, getMarker(marker), message, param);
  }

  @Override
  public void info(Marker marker, String message, Object... params) {
    logger.logIfEnabled(FQCN, Level.INFO, getMarker(marker), message, params);
  }

  @Override
  public void info(Marker marker, String message, Throwable cause) {
    logger.logIfEnabled(FQCN, Level.INFO, getMarker(marker), message, cause);
  }

  @Override
  public void info(Marker marker, String message, Object param, Throwable cause) {
    logger.logIfEnabled(FQCN, Level.INFO, getMarker(marker), message, param, cause);
  }

  @Override
  public void info(Marker marker, String message, Object[] params, Throwable cause) {
    logger.logIfEnabled(FQCN, Level.INFO, getMarker(marker), message, params, cause);
  }

  @Override
  public boolean isWarnEnabled() {
    return logger.isEnabled(Level.WARN, null, null);
  }

  @Override
  public boolean isWarnEnabled(Marker marker) {
    return logger.isEnabled(Level.WARN, getMarker(marker), null);
  }

  @Override
  public void warn(String message) {
    logger.logIfEnabled(FQCN, Level.WARN, null, message);
  }

  @Override
  public void warn(String message, Object param) {
    logger.logIfEnabled(FQCN, Level.WARN, null, message, param);
  }

  @Override
  public void warn(String message, Object... params) {
    logger.logIfEnabled(FQCN, Level.WARN, null, message, params);
  }

  @Override
  public void warn(String message, Throwable cause) {
    logger.logIfEnabled(FQCN, Level.WARN, null, message, cause);
  }

  @Override
  public void warn(String message, Object param, Throwable cause) {
    logger.logIfEnabled(FQCN, Level.WARN, null, message, param, cause);
  }

  @Override
  public void warn(String message, Object[] params, Throwable cause) {
    logger.logIfEnabled(FQCN, Level.WARN, null, message, params, cause);
  }

  @Override
  public void warn(Marker marker, String message) {
    logger.logIfEnabled(FQCN, Level.WARN, getMarker(marker), message);
  }

  @Override
  public void warn(Marker marker, String message, Object param) {
    logger.logIfEnabled(FQCN, Level.WARN, getMarker(marker), message, param);
  }

  @Override
  public void warn(Marker marker, String message, Object... params) {
    logger.logIfEnabled(FQCN, Level.WARN, getMarker(marker), message, params);
  }

  @Override
  public void warn(Marker marker, String message, Throwable cause) {
    logger.logIfEnabled(FQCN, Level.WARN, getMarker(marker), message, cause);
  }

  @Override
  public void warn(Marker marker, String message, Object param, Throwable cause) {
    logger.logIfEnabled(FQCN, Level.WARN, getMarker(marker), message, param, cause);
  }

  @Override
  public void warn(Marker marker, String message, Object[] params, Throwable cause) {
    logger.logIfEnabled(FQCN, Level.WARN, getMarker(marker), message, params, cause);
  }

  @Override
  public boolean isErrorEnabled() {
    return logger.isEnabled(Level.ERROR, null, null);
  }

  @Override
  public boolean isErrorEnabled(Marker marker) {
    return logger.isEnabled(Level.ERROR, getMarker(marker), null);
  }

  @Override
  public void error(String message) {
    logger.logIfEnabled(FQCN, Level.ERROR, null, message);
  }

  @Override
  public void error(String message, Object param) {
    logger.logIfEnabled(FQCN, Level.ERROR, null, message, param);
  }

  @Override
  public void error(String message, Object... params) {
    logger.logIfEnabled(FQCN, Level.ERROR, null, message, params);
  }

  @Override
  public void error(String message, Throwable cause) {
    logger.logIfEnabled(FQCN, Level.ERROR, null, message, cause);
  }

  @Override
  public void error(String message, Object param, Throwable cause) {
    logger.logIfEnabled(FQCN, Level.ERROR, null, message, param, cause);
  }

  @Override
  public void error(String message, Object[] params, Throwable cause) {
    logger.logIfEnabled(FQCN, Level.ERROR, null, message, params, cause);
  }

  @Override
  public void error(Marker marker, String message) {
    logger.logIfEnabled(FQCN, Level.ERROR, getMarker(marker), message);
  }

  @Override
  public void error(Marker marker, String message, Object param) {
    logger.logIfEnabled(FQCN, Level.ERROR, getMarker(marker), message, param);
  }

  @Override
  public void error(Marker marker, String message, Object... params) {
    logger.logIfEnabled(FQCN, Level.ERROR, getMarker(marker), message, params);
  }

  @Override
  public void error(Marker marker, String message, Throwable cause) {
    logger.logIfEnabled(FQCN, Level.ERROR, getMarker(marker), message, cause);
  }

  @Override
  public void error(Marker marker, String message, Object param, Throwable cause) {
    logger.logIfEnabled(FQCN, Level.ERROR, getMarker(marker), message, param, cause);
  }

  @Override
  public void error(Marker marker, String message, Object[] params, Throwable cause) {
    logger.logIfEnabled(FQCN, Level.ERROR, getMarker(marker), message, params, cause);
  }

  @Override
  public boolean isFatalEnabled() {
    return logger.isEnabled(Level.FATAL, null, null);
  }

  @Override
  public boolean isFatalEnabled(Marker marker) {
    return logger.isEnabled(Level.FATAL, getMarker(marker), null);
  }

  @Override
  public void fatal(String message) {
    logger.logIfEnabled(FQCN, Level.FATAL, null, message);
  }

  @Override
  public void fatal(String message, Object param) {
    logger.logIfEnabled(FQCN, Level.FATAL, null, message, param);
  }

  @Override
  public void fatal(String message, Object... params) {
    logger.logIfEnabled(FQCN, Level.FATAL, null, message, params);
  }

  @Override
  public void fatal(String message, Throwable cause) {
    logger.logIfEnabled(FQCN, Level.FATAL, null, message, cause);
  }

  @Override
  public void fatal(String message, Object param, Throwable cause) {
    logger.logIfEnabled(FQCN, Level.FATAL, null, message, param, cause);
  }

  @Override
  public void fatal(String message, Object[] params, Throwable cause) {
    logger.logIfEnabled(FQCN, Level.FATAL, null, message, params, cause);
  }

  @Override
  public void fatal(Marker marker, String message) {
    logger.logIfEnabled(FQCN, Level.FATAL, getMarker(marker), message);
  }

  @Override
  public void fatal(Marker marker, String message, Object param) {
    logger.logIfEnabled(FQCN, Level.FATAL, getMarker(marker), message, param);
  }

  @Override
  public void fatal(Marker marker, String message, Object... params) {
    logger.logIfEnabled(FQCN, Level.FATAL, getMarker(marker), message, params);
  }

  @Override
  public void fatal(Marker marker, String message, Throwable cause) {
    logger.logIfEnabled(FQCN, Level.FATAL, getMarker(marker), message, cause);
  }

  @Override
  public void fatal(Marker marker, String message, Object param, Throwable cause) {
    logger.logIfEnabled(FQCN, Level.FATAL, getMarker(marker), message, param, cause);
  }

  @Override
  public void fatal(Marker marker, String message, Object[] params, Throwable cause) {
    logger.logIfEnabled(FQCN, Level.FATAL, getMarker(marker), message, params, cause);
  }

  @Override
  public void log(String fqcn, org.seppiko.glf.api.Level level, Marker marker, String message,
      Object[] params, Throwable cause) {
    final Level log4jLevel = getLevel(level);
    final org.apache.logging.log4j.Marker log4jMarker = getMarker(marker);

    if (!logger.isEnabled(log4jLevel, log4jMarker, message, params)) {
      return;
    }
    final Message msg;
//    if (CONVERTER != null && eventLogger && marker != null && marker.contains(EVENT_MARKER)) {
//      msg = CONVERTER.convertEvent(message, params, cause);
//    } else
    if (params == null) {
      msg = new SimpleMessage(message);
    } else {
      msg = new ParameterizedMessage(message, params, cause);
      if (cause != null) {
        cause = msg.getThrowable();
      }
    }
    logger.logMessage(fqcn, log4jLevel, log4jMarker, msg, cause);
  }

  private static org.apache.logging.log4j.Marker getMarker(final Marker marker) {
    if (marker == null) {
      return null;
    } else if (marker instanceof Log4j2Marker) {
      return ((Log4j2Marker) marker).getLog4j2Marker();
    } else {
      final Log4j2MarkerFactory factory = (Log4j2MarkerFactory) StaticMarkerBinder.SINGLETON.getMarkerFactory();
      return (org.apache.logging.log4j.Marker) factory.getMarker(marker);
    }
  }

  private static Level getLevel(org.seppiko.glf.api.Level level) {
    switch (level) {
      case TRACE:
        return Level.TRACE;
      case DEBUG:
        return Level.DEBUG;
      case INFO:
        return Level.INFO;
      case WARN:
        return Level.WARN;
      case ERROR:
        return Level.ERROR;
    }
    return Level.FATAL;
  }

}
