/*
 * Copyright 2023 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.seppiko.glf.scl;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Objects;
import java.util.function.Supplier;
import org.seppiko.commons.logging.Logging;
import org.seppiko.commons.logging.LoggingBuilder;
import org.seppiko.glf.api.Logger;

/**
 * Commons Logging over GLF
 *
 * @author Leonard Woo
 */
public class GLFLogging implements Logging {

  private final Logger logger;

  GLFLogging(Logger logger) {
    this.logger = logger;
  }

  private Object[] getObjects(Supplier<?>[] suppliers) {
    ArrayList<Object> tlist = new ArrayList<>();
    Arrays.stream(suppliers)
        .filter(Objects::nonNull)
        .forEach(supplier -> {
      tlist.add(supplier.get());
    });
    return tlist.toArray(Object[]::new);
  }

  @Override
  public String getName() {
    return logger.getName();
  }

  /** {@inheritDoc} */
  @Override
  public boolean isEnable(int level) {
    return switch (level) {
      case LEVEL_TRACE_INT -> logger.isTraceEnabled();
      case LEVEL_DEBUG_INT -> logger.isDebugEnabled();
      case LEVEL_INFO_INT -> logger.isInfoEnabled();
      case LEVEL_WARN_INT -> logger.isWarnEnabled();
      case LEVEL_ERROR_INT -> logger.isErrorEnabled();
      default -> throw new IllegalArgumentException("");
    };
  }

  /** {@inheritDoc} */
  @Override
  public void log(int level, CharSequence message) {
    log(level, message, null);
  }

  /** {@inheritDoc} */
  @Override
  public void log(int level, CharSequence message, Throwable cause) {
    log(level, message, cause, (Supplier<?>) null);
  }

  /** {@inheritDoc} */
  @Override
  public void log(int level, CharSequence message, Throwable cause, Supplier<?>... paramSuppliers) {
    switch (level) {
      case LEVEL_TRACE_INT -> {
        logger.trace(message.toString(), cause, getObjects(paramSuppliers));
      }
      case LEVEL_DEBUG_INT -> {
        logger.debug(message.toString(), cause, getObjects(paramSuppliers));
      }
      case LEVEL_INFO_INT -> {
        logger.info(message.toString(), cause, getObjects(paramSuppliers));
      }
      case LEVEL_WARN_INT -> {
        logger.warn(message.toString(), cause, getObjects(paramSuppliers));
      }
      case LEVEL_ERROR_INT -> {
        logger.error(message.toString(), cause, getObjects(paramSuppliers));
      }
      case LEVEL_FATAL_INT -> {
        logger.fatal(message.toString(), cause, getObjects(paramSuppliers));
      }
      default -> throw new IllegalArgumentException("Not found level");
    }
  }

  /** {@inheritDoc} */
  @Override
  public void log(int level, CharSequence message, Throwable cause, Object... params) {
    switch (level) {
      case LEVEL_TRACE_INT -> {
        logger.trace(message.toString(), cause, params);
      }
      case LEVEL_DEBUG_INT -> {
        logger.debug(message.toString(), cause, params);
      }
      case LEVEL_INFO_INT -> {
        logger.info(message.toString(), cause, params);
      }
      case LEVEL_WARN_INT -> {
        logger.warn(message.toString(), cause, params);
      }
      case LEVEL_ERROR_INT -> {
        logger.error(message.toString(), cause, params);
      }
      case LEVEL_FATAL_INT -> {
        logger.fatal(message.toString(), cause, params);
      }
      default -> throw new IllegalArgumentException("Not found level");
    }
  }

  /** {@inheritDoc} */
  @Override
  public boolean isTraceEnable() {
    return isEnable(LEVEL_TRACE_INT);
  }

  /** {@inheritDoc} */
  @Override
  public void trace(CharSequence message) {
    log(LEVEL_TRACE_INT, message);
  }

  /** {@inheritDoc} */
  @Override
  public void trace(CharSequence message, Throwable cause) {
    log(LEVEL_TRACE_INT, message, cause);
  }

  /** {@inheritDoc} */
  @Override
  public void trace(CharSequence message, Throwable cause, Supplier<?>... paramsSupplier) {
    log(LEVEL_TRACE_INT, message, cause, paramsSupplier);
  }

  /** {@inheritDoc} */
  @Override
  public LoggingBuilder atTrace() {
    return new GLFLoggingBuilder(this, LEVEL_TRACE_INT);
  }

  /** {@inheritDoc} */
  @Override
  public boolean isDebugEnable() {
    return isEnable(LEVEL_DEBUG_INT);
  }

  /** {@inheritDoc} */
  @Override
  public void debug(CharSequence message) {
    log(LEVEL_DEBUG_INT, message);
  }

  /** {@inheritDoc} */
  @Override
  public void debug(CharSequence message, Throwable cause) {
    log(LEVEL_DEBUG_INT, message, cause);
  }

  /** {@inheritDoc} */
  @Override
  public void debug(CharSequence message, Throwable cause, Supplier<?>... paramsSupplier) {
    log(LEVEL_DEBUG_INT, message, cause, paramsSupplier);
  }

  /** {@inheritDoc} */
  @Override
  public LoggingBuilder atDebug() {
    return new GLFLoggingBuilder(this, LEVEL_DEBUG_INT);
  }

  /** {@inheritDoc} */
  @Override
  public boolean isInfoEnable() {
    return isEnable(LEVEL_INFO_INT);
  }

  /** {@inheritDoc} */
  @Override
  public void info(CharSequence message) {
    log(LEVEL_INFO_INT, message);
  }

  /** {@inheritDoc} */
  @Override
  public void info(CharSequence message, Throwable cause) {
    log(LEVEL_INFO_INT, message, cause);
  }

  /** {@inheritDoc} */
  @Override
  public void info(CharSequence message, Throwable cause, Supplier<?>... paramsSupplier) {
    log(LEVEL_INFO_INT, message, cause, paramsSupplier);
  }

  /** {@inheritDoc} */
  @Override
  public LoggingBuilder atInfo() {
    return new GLFLoggingBuilder(this, LEVEL_INFO_INT);
  }

  /** {@inheritDoc} */
  @Override
  public boolean isWarnEnable() {
    return isEnable(LEVEL_WARN_INT);
  }

  /** {@inheritDoc} */
  @Override
  public void warn(CharSequence message) {
    log(LEVEL_WARN_INT, message);
  }

  /** {@inheritDoc} */
  @Override
  public void warn(CharSequence message, Throwable cause) {
    log(LEVEL_WARN_INT, message, cause);
  }

  /** {@inheritDoc} */
  @Override
  public void warn(CharSequence message, Throwable cause, Supplier<?>... paramsSupplier) {
    log(LEVEL_WARN_INT, message, cause, paramsSupplier);
  }

  /** {@inheritDoc} */
  @Override
  public LoggingBuilder atWarn() {
    return new GLFLoggingBuilder(this, LEVEL_WARN_INT);
  }

  /** {@inheritDoc} */
  @Override
  public boolean isErrorEnable() {
    return isEnable(LEVEL_ERROR_INT);
  }

  /** {@inheritDoc} */
  @Override
  public void error(CharSequence message) {
    log(LEVEL_ERROR_INT, message);
  }

  /** {@inheritDoc} */
  @Override
  public void error(CharSequence message, Throwable cause) {
    log(LEVEL_ERROR_INT, message, cause);
  }

  /** {@inheritDoc} */
  @Override
  public void error(CharSequence message, Throwable cause, Supplier<?>... paramsSupplier) {
    log(LEVEL_ERROR_INT, message, cause, paramsSupplier);
  }

  /** {@inheritDoc} */
  @Override
  public LoggingBuilder atError() {
    return new GLFLoggingBuilder(this, LEVEL_ERROR_INT);
  }

  /** {@inheritDoc} */
  @Override
  public boolean isFatalEnable() {
    return isEnable(LEVEL_FATAL_INT);
  }

  /** {@inheritDoc} */
  @Override
  public void fatal(CharSequence message) {
    log(LEVEL_FATAL_INT, message);
  }

  /** {@inheritDoc} */
  @Override
  public void fatal(CharSequence message, Throwable cause) {
    log(LEVEL_FATAL_INT, message, cause);
  }

  /** {@inheritDoc} */
  @Override
  public void fatal(CharSequence message, Throwable cause, Supplier<?>... paramsSupplier) {
    log(LEVEL_FATAL_INT, message, cause, paramsSupplier);
  }

  /** {@inheritDoc} */
  @Override
  public LoggingBuilder atFatal() {
    return new GLFLoggingBuilder(this, LEVEL_FATAL_INT);
  }

}
