/*******************************************************************************
 * Copyright 2012 Roland Ewald
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *   http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 ******************************************************************************/
package sessl.omnetpp

import java.io.File
import java.io.FileWriter
import sessl.AbstractExperiment
import sessl.AfterSimTime
import sessl.AfterWallClockTime
import sessl.DisjunctiveStoppingCondition
import sessl.FixedNumber
import sessl.StoppingCondition
import java.net.URI
import sessl.util.MiscUtils
import sessl.util.ResultHandling

/**
 * Integrates OMNeT++ 4.2 (http://www.omnetpp.org/) by producing <code>omnetpp.ini</code> files.
 *  @author Roland Ewald
 */
class Experiment extends AbstractExperiment with OMNeTPPResultHandler {

  /** The default file header. */
  val fileHeader = "#\n# This file was automatically generated by SESSL. Do NOT edit manually.\n#\n\n[General]"

  /** The file to which the experiment configuration is written. */
  var expConfFileName = "omnetpp.ini"

  /** The working directory. */
  private[this] var workingDirectory: Option[File] = None

  /** The file writer for the experiment configuration. */
  private[this] var fileWriter: Option[FileWriter] = None

  /** The executable file. */
  private[this] var executableFile: Option[File] = None

  /** The list of distinct parameter assignments. */
  private[this] var assignments = List[Map[String, Any]]()

  /** Generates the corresponding omnetpp.ini file. */
  def basicConfiguration(): Unit = {
    initializeExperimentConfigFile()
    
    writeComment("Basic Experiment Setup")
    configureModel()
    configureStopping()
    configureReplications()
    
    writeComment("Fixed Model Variables")
    configureFixedVariables()
    
    writeComment("Parameter Scan Setup")
    configureVariablesToScan()
  }

  /** Free file handle after the experiment is done. */
  override def finishExperiment() = {
    if (fileWriter.isDefined)
      fileWriter.get.close
  }

  /** The only valid way to set a model is by also giving the executable to simulate it.*/
  def model_=(execAndNetwork: (String, String)) {
    executableFile = Some(new File(execAndNetwork._1))
    modelLocation = Some(execAndNetwork._2)
  }

  override def model_=(modelLocation: String) = throw new UnsupportedOperationException
  override def model_=(modelLocation: URI) = throw new UnsupportedOperationException

  /** Initializes the experiment configuration file. */
  def initializeExperimentConfigFile(): Unit = {
    checkExecutable()
    initializeConfFileWriter()
  }

  /** Checks whether given executable and model file can be used. */
  private[this] def checkExecutable(): Unit = {
    require(executableFile.isDefined && modelLocation.isDefined, "Simulator executable and model need to be defined. Use model = (\"simulator/executable\" -> \"Network\")")
    require({ val e = executableFile.get; e.exists && e.isFile && e.canExecute }, "Simulator execitable '" + executableFile.get.getAbsolutePath + "' not found or not executable.")
  }

  /** Initialize experiment configuration file. */
  private[this] def initializeConfFileWriter() = {
    workingDirectory = Some(new File(executableFile.get.getAbsolutePath).getParentFile)
    val expConfFile = new File(workingDirectory.get.getAbsolutePath + File.separator + expConfFileName)
    if (expConfFile.exists)
      require(expConfFile.delete, "Could not delete previous experiment configuration file: '" + expConfFile.getAbsolutePath + "'.")
    fileWriter = Some(new FileWriter(expConfFile, false))
    write(fileHeader)
  }

  /** Configure the model to be simulated. */
  def configureModel(): Unit = {
    write("network", modelLocation.get)
  }

  /** Configure stop condition. */
  def configureStopping(): Unit = {
    writeStoppingCondition(checkAndGetStoppingCondition())
  }

  /** Writes a stopping condition into the next line. */
  def writeStoppingCondition(c: StoppingCondition): Unit = {
    c match {
      case st: AfterSimTime => write("sim-time-limit", st.asMilliSecondsOrUnitless + "ms")
      case wct: AfterWallClockTime => write("cpu-time-limit", wct.asMilliSecondsOrUnitless + "ms")
      case or: DisjunctiveStoppingCondition => {
        writeStoppingCondition(or.left)
        writeStoppingCondition(or.right)
      }
      case x => throw new IllegalArgumentException("Stopping criterion '" + c + "' not supported.")
    }
  }

  /** Configures all fixed variables. */
  def configureFixedVariables(): Unit = {
    fixedVariables.foreach(writeVariableDefinition)
  }

  /** Writes a variable definition. */
  def writeVariableDefinition(element: (String, _)): Unit = {
    write(element._1, variableValueAsString(element._2))
  }

  /** Returns string representation of a value. */
  def variableValueAsString(value: Any): String = {
    value match {
      case str: String => "\"" + str + "\""
      case x => x.toString()
    }
  }

  /** Configures all variables for which the values shall be scanned. */
  def configureVariablesToScan(): Unit = {

    assignments = createVariableSetups()

    //Merge all values to be assigned to a variable to a single list 
    val varValuesLists = assignments.foldLeft(Map[String, List[Any]]()) {
      (m1, m2) => m2.map(x => (x._1, x._2 :: m1.getOrElse(x._1, List())))
    }.map(x => (x._1, x._2.reverse))

    //Define all variables to be set up in parallel
    for (varValues <- varValuesLists.zipWithIndex) {
      write(varValues._1._1, "${v" + varValues._2 + "= " + varValues._1._2.mkString(",") + { if (varValues._2 == 0) "}" else " ! v0}" })
    }
  }

  /** Configures the number of replications. */
  def configureReplications(): Unit = {
    checkAndGetReplicationCondition() match {
      case fixed: FixedNumber => write("repeat", fixed.replications.toString)
      case x => throw new IllegalArgumentException("Only a fixed number of replications is supported.")
    }
  }

  /** Executes OMNeT++ experiment by repeatedly execution the executable file. */
  def executeExperiment() = {

    val numOfReps = fixedReplications.get
    val numOfJobs = assignments.length

    //TODO: Maybe add *parallel* replications via collections ?
    for (assignmentId <- Range(0, numOfJobs)) {
      for (repNum <- Range(0, numOfReps)) {
        val runId = assignmentId * numOfReps + repNum
        addAssignmentForRun(runId, assignmentId, assignments(assignmentId).toList)
        OMNeTPPExecutor.execute(workingDirectory.get, executableFile.get, runId)
        considerResults(runId, workingDirectory.get);
        runDone(runId)
      }
      replicationsDone(assignmentId)
    }
    checkResultHandlingCorrectness("considerResults(...)")
    experimentDone()
  }

  /** Alias for brevity. */
  private[this] def write(content: String): Unit = {
    fileWriter.get.write(content + '\n')
    fileWriter.get.flush
  }

  /** Alias for brevity. */
  private[omnetpp] def write(key: String, value: String): Unit = write(key + " = " + value)

  /** Writes comment. */
  private[omnetpp] def writeComment(comment: String) = write("\n# " + comment)
}
