/*
 * Copyright 2017 University of Rostock
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package sessl.pssalib

import java.io.File
import java.nio.file._
import java.util.concurrent.{ExecutorService, Executors}

import sessl._

import scala.collection.mutable
import scala.concurrent.{Await, ExecutionContext, Future}

/**
  * A PSSALib experiment gets executed by invoking the native binary (which must be installed separately).
  * To configure the model with parameter values, the SBML model file is copied and modified before it becomes the input for the simulator.
  * A temporary directory is created for these model file copies.
  * The individual simulation runs are started in Futures to enable concurrency.
  *
  * @author Tom Warnke
  */
class Experiment extends AbstractExperiment with DynamicSimulationRuns with SBMLFileManipulation {

  private val tmpDir = new File(Files.createTempDirectory("tmp-pssalib-exp").toUri)
  sys.addShutdownHook(util.Files.deleteFileOrDirectory(tmpDir))

  protected def dirForRun(runId: Int) = new File(tmpDir, s"run-$runId")

  private[this] val futures = mutable.ListBuffer.empty[Future[Unit]]

  protected lazy val sim: PSSALibSimulator = simulator.asInstanceOf[PSSALibSimulator]

  override protected def basicConfiguration(): Unit = {}

  override protected[sessl] def executeExperiment(): Unit = {
    for (assignmentId <- variableAssignments.indices) {
      issueRuns(assignmentId)
    }
    Await.ready(Future.sequence(futures), scala.concurrent.duration.Duration.Inf)
    executor.shutdownNow()
  }

  override protected def variableAssignments: List[Map[String, Any]] =
    createVariableSetups().map(assignment => assignment ++ fixedVariables)

  override protected def onFinishedExperiment(): Unit = {
    logger.info("All jobs finished.")
  }

  override protected def startSimulationRun(runId: Int, assignmentId: Int): Boolean = {

    val f = Future {

      val runDir = dirForRun(runId)
      runDir.mkdir()

      val assignment = variableAssignments(assignmentId).collect {
        case (name, value: Double) => (name, value)
      }

      val originalModel = readFile(model)
      overrideParamValues(originalModel, assignment)
      val modelFile = writeFile(originalModel, runDir)

      import scala.sys.process._

      val logger = ProcessLogger(new File(runDir, "run.log"))

      val process = Process(
        Seq[String](
          sim.simulatorBinary,
          "--sbml-file", modelFile.getAbsolutePath,
          "--methods",  sim.argument,
          "--num-samples", "1",
          "--tend", stopTime.toLong.toString,
          "--output-path", runDir.getAbsolutePath) ++
        additionalOptions,
        sim.pathFile)

      process.!<(logger) // blocking process execution
      logger.flush()

      onFinishedRun(runId)
    }

    futures += f

    true
  }

  protected val executor: ExecutorService = Executors.newSingleThreadExecutor()

  implicit private val execContext: ExecutionContext =
    ExecutionContext.fromExecutor(executor)

  protected def additionalOptions: Seq[String] = Seq.empty

}

