package sila_java.interoperability.client;

import io.grpc.ManagedChannel;
import lombok.SneakyThrows;
import lombok.extern.slf4j.Slf4j;
import sila_java.interoperability.client.features.*;
import sila_java.library.core.sila.clients.ChannelFactory;
import sila_java.library.manager.ServerFinder;
import sila_java.library.manager.ServerManager;
import sila_java.library.manager.models.Server;

import java.time.Duration;
import java.util.concurrent.TimeUnit;

@Slf4j
public class Client {
    public static final String SERVER_TYPE = "InteroperabilitySiLAServer";
    private final ManagedChannel serviceChannel;
    private final UnobservablePropertyTest unobservablePropertyTest;
    private final UnobservableCommandTest unobservableCommandTest;
    private final ObservablePropertyTest observablePropertyTest;
    private final ObservableCommandTest observableCommandTest;
    private final MetadataProvider metadataProvider;
    private final MetadataConsumerTest metadataConsumerTest;
    private final ErrorHandlingTest errorHandlingTest;
    private final BinaryTransferTest binaryTransferTest;

    public Client(final ManagedChannel serviceChannel) {
        this.serviceChannel = serviceChannel;
        this.unobservablePropertyTest = new UnobservablePropertyTest(serviceChannel);
        this.unobservableCommandTest = new UnobservableCommandTest(serviceChannel);
        this.observablePropertyTest = new ObservablePropertyTest(serviceChannel);
        this.observableCommandTest = new ObservableCommandTest(serviceChannel);
        this.metadataProvider = new MetadataProvider(serviceChannel);
        this.metadataConsumerTest = new MetadataConsumerTest(serviceChannel);
        this.errorHandlingTest = new ErrorHandlingTest(serviceChannel);
        this.binaryTransferTest = new BinaryTransferTest(serviceChannel);
    }

    /**
     * Simple Client that stops after using the GreetingProvider Feature
     */
    @SneakyThrows
    public static void main(String[] args) {
        try (final ServerManager serverManager = ServerManager.getInstance()) {

            // Create Manager for clients and start discovery
            final Server server = ServerFinder
                    .filterBy(ServerFinder.Filter.type(SERVER_TYPE))
                    .scanAndFindOne(Duration.ofMinutes(1))
                    .orElseThrow(() -> new RuntimeException("No " + SERVER_TYPE + " found within time"));

            log.info("Found Server!");

            final ManagedChannel serviceChannel = ChannelFactory.getTLSEncryptedChannel(
                    server.getHost(), server.getPort(), server.getCertificateAuthority()
            );
            try {
                final Client client = new Client(serviceChannel);
                client.runUnobservablePropertyTest();
                client.runUnobservableCommandTest();
                client.runObservablePropertyTest();
                client.runObservableCommandTest();
                client.runMetadataProvider();
                client.runMetadataConsumerTest();
                client.runErrorHandlingTest();
                client.runBinaryTransferTest();
            } finally {
                serviceChannel.shutdown().awaitTermination(5, TimeUnit.SECONDS);
            }
        }
    }

    public void runBinaryTransferTest() {
        binaryTransferTest.getFCPAffectedByMetadataString();
        binaryTransferTest.binaryValueDirectly();
        binaryTransferTest.echoBinariesObservably();
        binaryTransferTest.echoBinaryAndMetadataString();
        binaryTransferTest.binaryValueDirectly();
    }

    public void runErrorHandlingTest() {
        errorHandlingTest.raiseDefinedExecutionError();
        errorHandlingTest.raiseDefinedExecutionErrorObservably();
        errorHandlingTest.raiseUndefinedExecutionError();
        errorHandlingTest.raiseUndefinedExecutionErrorObservably();
        errorHandlingTest.raiseDefinedExecutionErrorOnGet();
        errorHandlingTest.raiseDefinedExecutionErrorOnSubscribe();
        errorHandlingTest.raiseUndefinedExecutionErrorOnGet();
        errorHandlingTest.raiseUndefinedExecutionErrorOnSubscribe();
        errorHandlingTest.raiseDefinedExecutionErrorAfterValueWasSent();
        errorHandlingTest.raiseUndefinedExecutionErrorAfterValueWasSent();
    }

    public void runMetadataConsumerTest() {
        metadataConsumerTest.echoStringMetadata();
        metadataConsumerTest.unpackMetadata();
    }

    public void runMetadataProvider() {
        metadataProvider.getFCPAffectedByMetadataStringMetadata();
        metadataProvider.getFCPAffectedByMetadataStringMetadata();
    }

    public void runObservableCommandTest() {
        observableCommandTest.count(0.5, 5);
        observableCommandTest.echo(0.5, 42);
    }

    public void runObservablePropertyTest() {
        observablePropertyTest.subscribeFixedValue();
        observablePropertyTest.subscribeAlternating();
        observablePropertyTest.subscribeEditable();
    }

    public void runUnobservableCommandTest() {
        unobservableCommandTest.commandWithParametersAndResponses();
        unobservableCommandTest.convertIntegerToString(42);
        unobservableCommandTest.joinIntegerAndString(42, "1337");
        unobservableCommandTest.splitStringAfterFirstChar("Sila");
    }
    public void runUnobservablePropertyTest() {
        unobservablePropertyTest.answerToEverything();
        unobservablePropertyTest.secondsSince1970();
    }
}