package sila_java.interoperability.server;

import java.io.IOException;
import java.sql.SQLException;
import java.util.Objects;

import lombok.NonNull;
import lombok.extern.slf4j.Slf4j;
import sila_java.interoperability.server.features.*;
import sila_java.library.server_base.SiLAServer;
import sila_java.library.server_base.binary_transfer.database.BinaryDatabase;
import sila_java.library.server_base.binary_transfer.database.impl.H2BinaryDatabase;
import sila_java.library.server_base.config.IServerConfigWrapper;
import sila_java.library.server_base.identification.ServerInformation;
import sila_java.library.server_base.utils.ArgumentHelper;
import  sila_java.interoperability.repository.EmptyClass;

import static sila_java.library.core.utils.FileUtils.getFileContent;

/**
 * Example minimal SiLA Server
 */
@Slf4j
public class Server implements AutoCloseable {
    public static final String SERVER_TYPE = "InteroperabilitySiLAServer";

    private final BinaryDatabase binaryDatabase;

    private final SiLAServer server;
    public static final ServerInformation serverInfo = new ServerInformation(
            SERVER_TYPE,
            "Interoperability SiLA Server",
            "https://www.sila-standard.org",
            "0.1"
    );
    private final IServerConfigWrapper configuration;

    /**
     * Application Class using command line arguments
     *
     * @param argumentHelper Custom Argument Helper
     */
    Server(@NonNull final ArgumentHelper argumentHelper) {
        try {
            final SiLAServer.Builder builder = SiLAServer.Builder.newBuilder(serverInfo);

            builder.withPersistentConfig(argumentHelper.getConfigFile().isPresent());
            argumentHelper.getConfigFile().ifPresent(builder::withPersistentConfigFile);

            this.configuration = builder.getNewServerConfigurationWrapper();
            builder.withCustomConfigWrapperProvider((configPath, serverConfiguration) -> this.configuration);
            this.binaryDatabase = new H2BinaryDatabase(this.configuration.getCacheConfig().getUuid());
            builder.withCustomBinaryDatabaseProvider((uuid) -> this.binaryDatabase);
            builder.withMetadataExtractingInterceptor();
            builder.withBinaryTransferSupport();
            builder.withUnsafeCommunication(true);

            builder.withPersistentTLS(
                    argumentHelper.getPrivateKeyFile(),
                    argumentHelper.getCertificateFile(),
                    argumentHelper.getCertificatePassword()
            );

            if (argumentHelper.useUnsafeCommunication()) {
                builder.withUnsafeCommunication(true);
            }

            argumentHelper.getHost().ifPresent(builder::withHost);
            argumentHelper.getPort().ifPresent(builder::withPort);
            argumentHelper.getInterface().ifPresent(builder::withNetworkInterface);
            builder.withDiscovery(argumentHelper.hasNetworkDiscovery());

            final String featuresPath = "/sila_interoperability/communication-tests/sila2_interop_communication_tester/resources/fdl/";

            builder.addFeature(
                    getFileContent(
                            Objects.requireNonNull(
                                    EmptyClass.class.getResourceAsStream(featuresPath + "UnobservablePropertyTest.sila.xml")
                            )
                    ),
                    new UnobservablePropertyTest()
            );
            builder.addFeature(
                    getFileContent(
                            Objects.requireNonNull(
                                    EmptyClass.class.getResourceAsStream(featuresPath + "UnobservableCommandTest.sila.xml")
                            )
                    ),
                    new UnobservableCommandTest()
            );
            builder.addFeature(
                    getFileContent(
                            Objects.requireNonNull(
                                    EmptyClass.class.getResourceAsStream(featuresPath + "ObservablePropertyTest.sila.xml")
                            )
                    ),
                    new ObservablePropertyTest()
            );
            builder.addFeature(
                    getFileContent(
                            Objects.requireNonNull(
                                    EmptyClass.class.getResourceAsStream(featuresPath + "ObservableCommandTest.sila.xml")
                            )
                    ),
                    new ObservableCommandTest()
            );
            builder.addFeature(
                    getFileContent(
                            Objects.requireNonNull(
                                    EmptyClass.class.getResourceAsStream(featuresPath + "MetadataProvider.sila.xml")
                            )
                    ),
                    new MetadataProvider()
            );
            builder.addFeature(
                    getFileContent(
                            Objects.requireNonNull(
                                    EmptyClass.class.getResourceAsStream(featuresPath + "MetadataConsumerTest.sila.xml")
                            )
                    ),
                    new MetadataConsumerTest()
            );
            builder.addFeature(
                    getFileContent(
                            Objects.requireNonNull(
                                    EmptyClass.class.getResourceAsStream(featuresPath + "ErrorHandlingTest.sila.xml")
                            )
                    ),
                    new ErrorHandlingTest()
            );
            builder.addFeature(
                    getFileContent(
                            Objects.requireNonNull(
                                    EmptyClass.class.getResourceAsStream(featuresPath + "BinaryTransferTest.sila.xml")
                            )
                    ),
                    new BinaryTransferTest(this.binaryDatabase)
            );

            builder.addFCPAffectedByMetadata(
                    "org.silastandard/test/BinaryTransferTest/v1/Command/EchoBinaryAndMetadataString",
                    "org.silastandard/test/BinaryTransferTest/v1/Metadata/String"
            );

            this.server = builder.start();
        } catch (IOException | SQLException e) {
            log.error("Something went wrong when building / starting server", e);
            throw new RuntimeException(e);
        }
    }

    @Override
    public void close() {
        this.server.close();
    }

    /**
     * Simple main function that starts the server and keeps it alive
     */
    public static void main(final String[] args) {
        final ArgumentHelper argumentHelper = new ArgumentHelper(args, SERVER_TYPE);
        final Server server = new Server(argumentHelper);
        log.info("To stop the server press CTRL + C.");
        server.server.blockUntilShutdown();
        log.info("termination complete.");
    }

}
