/*
 * Decompiled with CFR 0.152.
 */
package sila_java.library.communication;

import java.io.DataInputStream;
import java.io.DataOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.Arrays;
import java.util.LinkedList;
import java.util.List;
import java.util.concurrent.Callable;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.function.Supplier;
import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import lombok.NonNull;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import sila_java.library.communication.WelcomeFlusherFunction;
import sila_java.library.communication.socket.CommunicationSocket;

public class SynchronousCommunication
implements AutoCloseable {
    private static final Logger log = LoggerFactory.getLogger(SynchronousCommunication.class);
    private static final int INPUT_INTERVAL = 100;
    private final StringBuffer inputBuffer = new StringBuffer();
    private final ExecutorService executor = Executors.newCachedThreadPool();
    private OutputStream out = null;
    private InputStream in = null;
    private Thread inputThread = null;
    private boolean isUp = false;
    private final CommunicationSocket communicationSocket;
    private final String sendDelimiter;
    private final String receiveDelimiter;
    private final WelcomeFlusherFunction startupFunction;
    private final CheckResultInterface errorChecker;
    private HeartBeatAgent heartBeatAgent = null;

    private SynchronousCommunication(@NonNull CommunicationSocket communicationSocket, @NonNull String sendDelimiter, @NonNull String receiveDelimiter, @Nullable WelcomeFlusherFunction startupFunction, @Nullable CheckResultInterface errorChecker) {
        if (communicationSocket == null) {
            throw new NullPointerException("communicationSocket is marked non-null but is null");
        }
        if (sendDelimiter == null) {
            throw new NullPointerException("sendDelimiter is marked non-null but is null");
        }
        if (receiveDelimiter == null) {
            throw new NullPointerException("receiveDelimiter is marked non-null but is null");
        }
        this.communicationSocket = communicationSocket;
        this.sendDelimiter = sendDelimiter;
        this.receiveDelimiter = receiveDelimiter;
        this.startupFunction = startupFunction;
        this.errorChecker = errorChecker;
    }

    public void startHeartbeat(int samplingTime, @Nonnull Supplier<Boolean> connectionTester) {
        log.info("Starting HeartbeatAgent...");
        if (this.heartBeatAgent != null) {
            this.heartBeatAgent.stop();
            this.heartBeatAgent = null;
        }
        this.heartBeatAgent = new HeartBeatAgent(samplingTime, connectionTester);
        this.heartBeatAgent.start();
    }

    @Override
    public void close() {
        log.info("Shutting down synchronousCommunication");
        if (this.heartBeatAgent != null) {
            log.info("Stopping Heartbeat.");
            this.heartBeatAgent.stop();
        }
        try {
            this.closeStreams();
        }
        catch (IOException e) {
            log.warn("Wasn't able to close Synchronous Communication IO Streams: {}", (Object)e.getMessage());
        }
        if (!this.executor.isShutdown()) {
            this.executor.shutdownNow();
        }
    }

    public synchronized void open() throws IOException {
        if (this.isUp) {
            return;
        }
        log.info("Open/Re-open connection");
        this.communicationSocket.open();
        this.out = new DataOutputStream(this.communicationSocket.getOutputStream());
        this.in = new DataInputStream(this.communicationSocket.getInputStream());
        this.inputThread = new Thread(() -> {
            try {
                while (true) {
                    if (this.in.available() > 0) {
                        int numRead;
                        byte[] newData = new byte[this.in.available()];
                        if (newData.length != (numRead = this.in.read(newData, 0, newData.length))) {
                            throw new IOException("Buffer error in stream reading");
                        }
                        String character = new String(newData, 0, newData.length);
                        this.inputBuffer.append(character);
                        continue;
                    }
                    Thread.sleep(100L);
                }
            }
            catch (IOException | InterruptedException e) {
                log.info(e.getMessage());
                return;
            }
        });
        this.inputThread.start();
        if (this.startupFunction != null) {
            this.startupFunction.apply(this);
        }
        this.isUp = true;
    }

    private synchronized void closeStreams() throws IOException {
        log.warn("Closing connection");
        if (this.out != null) {
            this.out.close();
        }
        if (this.inputThread != null) {
            try {
                this.inputThread.interrupt();
                this.inputThread.join();
                this.inputThread = null;
            }
            catch (InterruptedException e) {
                throw new IOException(e.getMessage());
            }
        }
        if (this.in != null) {
            this.in.close();
        }
        this.communicationSocket.close();
        this.isUp = false;
    }

    public boolean isUp() {
        return this.isUp;
    }

    public synchronized String sendReceiveSingleLine(String msg, long timeout) throws IOException {
        List<String> response = this.sendReceive(msg, timeout);
        if (response.size() != 1) {
            throw new IllegalStateException("Only one response allowed in sendReceiveSingleLine");
        }
        return response.get(0);
    }

    public synchronized List<String> sendReceive(String msg, long timeout) throws IOException {
        this.inputBuffer.setLength(0);
        String finalMsg = msg + this.sendDelimiter;
        this.out.write(finalMsg.getBytes());
        this.out.flush();
        return this.read(timeout);
    }

    public synchronized List<String> read(long timeout) throws IOException {
        return this.read(timeout, this.receiveDelimiter);
    }

    public synchronized List<String> read(long timeout, @NonNull String receiveDelimiter) throws IOException {
        String result;
        if (receiveDelimiter == null) {
            throw new NullPointerException("receiveDelimiter is marked non-null but is null");
        }
        Callable<String> readTask = () -> {
            while (true) {
                String message = this.inputBuffer.toString();
                if (this.errorChecker != null) {
                    this.errorChecker.checkResult(message);
                }
                if (message.contains(receiveDelimiter)) {
                    log.debug("Message: {}", (Object)message);
                    int mark = this.inputBuffer.indexOf(receiveDelimiter);
                    this.inputBuffer.delete(0, mark + receiveDelimiter.length());
                    return message;
                }
                Thread.sleep(100L);
            }
        };
        Future<String> future = this.executor.submit(readTask);
        try {
            result = future.get(timeout, TimeUnit.MILLISECONDS);
        }
        catch (InterruptedException | TimeoutException ex) {
            this.inputBuffer.setLength(0);
            throw new IOException("Reading from serial buffer failed: " + ex.getClass().getName());
        }
        catch (ExecutionException e) {
            this.inputBuffer.setLength(0);
            throw new IOException("Executing serial command failed: " + e.getMessage());
        }
        finally {
            future.cancel(true);
        }
        return new LinkedList<String>(Arrays.asList(result.split("\\R+")));
    }

    private class HeartBeatAgent
    implements Runnable {
        private final Thread heartBeatThread = new Thread((Runnable)this, this.getClass().getName() + "_Thread");
        private final int samplingTime;
        private final Supplier<Boolean> connectionTester;

        HeartBeatAgent(@Nonnull int samplingTime, Supplier<Boolean> connectionTester) {
            this.samplingTime = samplingTime;
            this.connectionTester = connectionTester;
        }

        void start() {
            this.heartBeatThread.start();
        }

        void stop() {
            this.heartBeatThread.interrupt();
        }

        @Override
        public void run() {
            try {
                while (true) {
                    if (!SynchronousCommunication.this.isUp()) {
                        try {
                            SynchronousCommunication.this.open();
                        }
                        catch (IOException e) {
                            log.info("Serial Comm IO not possible: " + e.getMessage());
                            SynchronousCommunication.this.closeStreams();
                        }
                    } else if (!this.connectionTester.get().booleanValue()) {
                        log.info("Connection got lost");
                        SynchronousCommunication.this.closeStreams();
                    }
                    Thread.sleep(this.samplingTime);
                }
            }
            catch (IOException | InterruptedException e) {
                log.info("Driver heart beat interrupted.");
                try {
                    SynchronousCommunication.this.closeStreams();
                }
                catch (IOException e1) {
                    log.error("Error trying to closeStreams communication during Interrupt: {}", (Object)e1.getMessage());
                }
                Thread.currentThread().interrupt();
                return;
            }
        }
    }

    public static class Builder {
        private final CommunicationSocket communicationSocket;
        private String sendDelimiter = "\r\n";
        private String receiveDelimiter = "\r\n";
        private WelcomeFlusherFunction startupFunction = null;
        private CheckResultInterface errorChecker = null;

        public Builder(@NonNull CommunicationSocket communicationSocket) {
            if (communicationSocket == null) {
                throw new NullPointerException("communicationSocket is marked non-null but is null");
            }
            this.communicationSocket = communicationSocket;
        }

        public Builder withSendDelimiter(@NonNull String sendDelimiter) {
            if (sendDelimiter == null) {
                throw new NullPointerException("sendDelimiter is marked non-null but is null");
            }
            this.sendDelimiter = sendDelimiter;
            return this;
        }

        public Builder withReceiveDelimiter(@NonNull String receiveDelimiter) {
            if (receiveDelimiter == null) {
                throw new NullPointerException("receiveDelimiter is marked non-null but is null");
            }
            this.receiveDelimiter = receiveDelimiter;
            return this;
        }

        public Builder withStartupMessageFlusher(@NonNull WelcomeFlusherFunction startupFunction) {
            if (startupFunction == null) {
                throw new NullPointerException("startupFunction is marked non-null but is null");
            }
            this.startupFunction = startupFunction;
            return this;
        }

        public Builder withErrorChecker(CheckResultInterface errorCheckerFunction) {
            this.errorChecker = errorCheckerFunction;
            return this;
        }

        public SynchronousCommunication build() {
            return new SynchronousCommunication(this.communicationSocket, this.sendDelimiter, this.receiveDelimiter, this.startupFunction, this.errorChecker);
        }
    }

    public static interface CheckResultInterface {
        public void checkResult(String var1) throws IOException;
    }
}

