package sila_java.library.manager;

import com.google.protobuf.Descriptors;
import io.grpc.ManagedChannel;
import lombok.NonNull;
import lombok.extern.slf4j.Slf4j;
import lombok.val;
import sila_java.library.core.discovery.SiLAServerDiscovery;
import sila_java.library.core.mapping.MalformedSiLAFeature;
import sila_java.library.core.mapping.ProtoMapper;
import sila_java.library.manager.executor.ServerCallExecutor;
import sila_java.library.manager.models.Server;
import sila_java.library.manager.models.SiLACall;
import sila_java.library.manager.server_management.ServerHeartbeat;
import sila_java.library.manager.server_management.DiscoveryListener;
import sila_java.library.manager.server_management.ServerUtilities;
import sila_java.library.manager.server_management.SiLAConnection;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import java.security.KeyException;
import java.time.Duration;
import java.util.*;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.TimeoutException;
import java.util.stream.Collectors;

import static sila_java.library.core.asynchronous.MethodPoller.await;
import static sila_java.library.manager.server_management.ServerLoading.loadServer;

/**
 * Singleton Manager to manage SiLA Servers
 *
 * This Manager will keep a cache of all servers added either through discovery or via invoking
 * the add function by other means (e.g. manually). It will also keep track of their Status with a
 * Heartbeat.
 */
@Slf4j
public class SiLAManager {
    // Period of Heartbeat checking SiLA Status (note that pinging will also take time)
    private static final int HEARTBEAT = 1000; // [ms]
    private static SiLAManager Instance;
    private final Map<UUID, Server> silaServers = new ConcurrentHashMap<>();
    private final Map<UUID, SiLAConnection> silaConnections = new ConcurrentHashMap<>();
    private final List<SiLAServerListener> siLAServerListenerList = new CopyOnWriteArrayList<>();
    private final DiscoveryListener discoveryListener;
    private final ServerHeartbeat serverHeartbeat;
    private boolean discovery = false;

    /**
     * Listener for 3rd Party Applications
     */
    public interface SiLAServerListener {
        /**
         * Server was added to cache either manually or through discovery
         * @param serverUuid Server Unique Identifier
         * @param server SiLA server Model
         */
        void onSiLAServerAdded(UUID serverUuid, Server server);

        /**
         * Server was removed (can only happen manually)
         * @param serverUuid Server Unique Identifier
         */
        void onSiLAServerRemoved(UUID serverUuid);

        /**
         * Server was changed, e.g. the status from Online to Offline
         *
         * @implNote The application developer is expected to track Servers by his own
         *
         * @param serverUuid Server Unique Identifier
         */
        void onServerChange(UUID serverUuid);
    }

    public static SiLAManager getInstance() {
        synchronized (SiLAManager.class) {
            if(Instance == null) {
                Instance = new SiLAManager();
            }
        }
        return Instance;
    }

    private SiLAManager() {
        this.discoveryListener = new DiscoveryListener(this);
        SiLAServerDiscovery.getInstance().addListener(discoveryListener);
        // Heartbeat
        this.serverHeartbeat = new ServerHeartbeat(this, discoveryListener, HEARTBEAT);
    }

    /**
     * Initialise SiLA Manager with pre-defined SiLA Servers, e.g. from a persistence layer
     * @param silaServers Map containing all predefined SiLAServers
     */
    public void initialize(@NonNull final Map<UUID, Server> silaServers) {
        if (!this.silaServers.isEmpty() || !this.silaConnections.isEmpty()) {
            throw new IllegalStateException("SiLA Manager can only be initialised in an empty state");
        }
        log.info("Initializing SiLA Manager with " + silaServers.size() + " servers");

        this.silaServers.putAll(silaServers);

        this.silaServers.forEach((uuid, server) -> {
            final ManagedChannel managedChannel = ServerUtilities.buildNewChannel(server);
            this.silaConnections.put(uuid, createSiLAConnection(managedChannel, server));
        });
    }

    /**
     * Enable SiLA Discovery
     */
    public void startDiscovery() {
        if (!this.discovery) {
            log.debug("SiLA Manager started");
            this.discovery = true;
            SiLAServerDiscovery.getInstance().start();
        } else {
            log.info("An attempt was made to start the SiLA Manager discovery when it was already running");
        }
    }

    /**
     * Disable SiLA Discovery
     */
    public void stopDiscovery() {
        if (this.discovery) {
            log.debug("SiLA Manager stopped");
            this.discovery = false;
            SiLAServerDiscovery.getInstance().stop();
        } else {
            log.info("An attempt was made to stop the SiLA Manager discovery when it was already stopped");
        }
    }

    /**
     * Start Heartbeat to check Status of Servers
     */
    public void startHeartbeat() {
        this.serverHeartbeat.start();
    }

    /**
     * Stop Heartbeat
     */
    public void stopHeartbeat() {
        this.serverHeartbeat.stop();
    }

    /**
     * Add Server manually to cache
     *
     * @param host Host on which SiLA Server is exposed
     * @param port Port on which SiLA Server is exposed
     */
    public void addServer(@NonNull final String host, final int port) {
        this.addServer(host, port, null);
    }

    /**
     * Removing Server manually
     *
     * @param id Server Id used for referencing to the Server
     */
    public synchronized void removeServer(@NonNull final UUID id) {
        if (!silaServers.containsKey(id)) {
            log.info("Server with id: " + id + " already removed from cache.");
            return;
        }

        final UUID discoveryId = silaServers.get(id).getDiscoveryId();
        silaServers.remove(id);

        if (silaConnections.containsKey(id)) {
            silaConnections.get(id).close();
            silaConnections.remove(id);
        }

        if (discoveryId != null) {
            this.discoveryListener.removeFromCache(discoveryId);
        }

        log.info("[removeServer] removed serverId={}", id);
        // also update listeners
        siLAServerListenerList.forEach(listener -> listener.onSiLAServerRemoved(id));
    }

    /**
     * Add Server to Cache with Discovery Id
     *
     * @param host Host on which SiLA Server is exposed
     * @param port Port on which SiLA Server is exposed
     * @param discoveryId Discovery ID (provided by mDNS)
     */
    public void addServer(@NonNull final String host, final int port, @Nullable final UUID discoveryId) {
        // Create Server
        final Server server = new Server(discoveryId);
        server.setJoined(new Date());
        server.setHost(host);
        server.setPort(port);
        server.setStatus(Server.Status.INVALID);

        parseServer(server);
    }

    /**
     * Private Utility to parse Server into Map
     */
    private synchronized void parseServer(@NonNull final Server server) {
        // Establish Connection
        final ManagedChannel managedChannel = ServerUtilities.buildNewChannel(server);

        // Load SiLA Server
        loadServer(server, managedChannel);

        final UUID serverKeyUUID;
        if (server.getStatus().equals(Server.Status.INVALID)) {
            log.warn("Server saved in cache, but Invalid and with random UUID Key.");
            serverKeyUUID = UUID.randomUUID();
        } else {
            serverKeyUUID = server.getConfiguration().getUuid();

            log.info(
                    "[addServer] Resolved SiLA Server serverName={} on {}:{}",
                    server.getConfiguration().getName(),
                    server.getHost(),
                    server.getPort()
            );
        }

        // If same InstanceId, simply overwrite
        if (silaServers.containsKey(serverKeyUUID)) {
            log.warn("Server with id: " + serverKeyUUID + " was already in cache - overwritng.");
        }

        // Add Server and notify when successful resolving
        silaServers.put(serverKeyUUID, server);
        silaConnections.put(serverKeyUUID, createSiLAConnection(managedChannel, server));

        // Update Listener
        siLAServerListenerList.forEach(listener -> listener.onSiLAServerAdded(serverKeyUUID, server));
    }

    /**
     * Private Utility to create a managed SiLA Connection
     * @param managedChannel gRPC Channel to Server
     * @param server Server Description
     * @return Managed Connection
     */
    private SiLAConnection createSiLAConnection(
            @NonNull final ManagedChannel managedChannel,
            @NonNull final Server server
    ) {
        final SiLAConnection siLAConnection = new SiLAConnection(managedChannel);

        // Load gRPC Mappings
        final ProtoMapper protoMapper = new ProtoMapper();
        for (val feature : server.getFeatures()) {
            try {
                final String featureIdentifier = getFeatureId(feature.getIdentifier());
                final Descriptors.FileDescriptor protoFile = protoMapper.getProto(feature);
                siLAConnection.addFeatureService(featureIdentifier, protoFile.getServices().get(0));
            } catch (Descriptors.DescriptorValidationException | MalformedSiLAFeature e) {
                server.setStatus(Server.Status.INVALID);
                break;
            }
        }

        return siLAConnection;
    }

    /**
     * Set the Status of a Server
     * @param id Server Id
     * @param status Server Status
     */
    public void setServerStatus(@NonNull final UUID id, @NonNull final Server.Status status) {
        final Server server = this.silaServers.get(id);

        if (server == null) {
            log.warn("Server with id " + id.toString() + "doesnt exit.");
            return;
        }

        // Only Notify Change, if Change happened
        if (!server.getStatus().equals(status)) {
            server.setStatus(status);

            log.info("[changeServerStatus] change {} status to {}", id.toString(), status.toString());
            // also update listeners
            siLAServerListenerList.forEach(listener -> listener.onServerChange(id));
        }
    }

    /**
     * Get Map of managed SiLA Servers
     */
    public Map<UUID, Server> getSiLAServers() {
        return this.silaServers;
    }

    /**
     * Get Map of managed SiLA Connections
     */
    public Map<UUID, SiLAConnection> getSilaConnections() {return this.silaConnections;}

    /**
     * Get a List of Servers with Server Name
     * @param serverName SiLA Server Name (configurable)
     * @return List of Servers found
     */
    public List<Server> getSiLAServersByName(@NonNull final String serverName) {
        return this.silaServers
                .values()
                .stream()
                .filter(server -> server.getConfiguration().getName().equals(serverName))
                .collect(Collectors.toList());
    }

    /**
     * Add additional listener to retrieve SiLA Server information in-process
     */
    public void addSiLAServerListener(@Nonnull final SiLAServerListener siLAServerListener) {
        siLAServerListenerList.add(siLAServerListener);
    }

    public void removeSiLAServerListener(@Nonnull final SiLAServerListener siLAServerListener) {
        siLAServerListenerList.remove(siLAServerListener);
    }

    /**
     * Getting SiLA Call Executor from a SiLA Call defining it
     */
    public ServerCallExecutor newCallExecutor(@NonNull final SiLACall siLACall) throws KeyException {
        final SiLAConnection connection = silaConnections.get(siLACall.getServerId());
        return new ServerCallExecutor(connection, siLACall);
    }

    /**
     * Blocking call until online server is found in cache
     *
     * @param serverName SiLA Server Name (configurable)
     * @param timeOut    in ms
     * @throws TimeoutException when the server could not be found within the timeOut period
     */
    public void blockUntilServerFound(@NonNull final String serverName, final int timeOut) throws TimeoutException {
        try {
            await()
                    .atMost(Duration.ofMillis(timeOut))
                    .until(()->getSiLAServersByName(serverName)
                            .stream()
                            .filter(server -> server.getStatus().equals(Server.Status.ONLINE))
                            .anyMatch(server -> server.getConfiguration().getName().equals(serverName)))
                    .execute();
        } catch (ExecutionException e) {
            throw new RuntimeException(e.getCause());
        } catch (TimeoutException e) {
            throw new TimeoutException("Can not find SiLA Server " + serverName + " in " + timeOut + " ms");
        }

        log.info("Found SiLA Server " + serverName);
    }

    /**
     * This is a temporary hack to retrieve the Feature Identifier correctly, even if it's fully
     * qualified (delimited by '/')
     *
     * TODO: Find a cleaner way to do this
     *
     * @param featureIdentifier The full Feature Identifier
     * @return the subset of the feature identifier useful for gRPC Calls
     */
    private static String getFeatureId(@NonNull final String featureIdentifier) {
        final int lastIndexOf = featureIdentifier.lastIndexOf('/');
        if (lastIndexOf > 0 && lastIndexOf != featureIdentifier.length() - 1) {  // If fully qualified feature, convert to simple feature id
            return (featureIdentifier.substring(lastIndexOf + 1));
        }
        return (featureIdentifier);
    }
}
