package sila_java.library.manager.server_management;

import io.grpc.ManagedChannel;
import io.grpc.StatusRuntimeException;
import lombok.AccessLevel;
import lombok.NoArgsConstructor;
import lombok.NonNull;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.io.input.CharSequenceReader;
import sila2.org.silastandard.core.silaservice.v1.SiLAServiceGrpc;
import sila2.org.silastandard.core.silaservice.v1.SiLAServiceOuterClass;
import sila_java.library.core.models.Feature;
import sila_java.library.manager.models.Server;
import sila_java.library.server_base.config.ServerConfiguration;
import sila_java.library.server_base.identification.ServerInformation;

import java.util.UUID;
import java.util.concurrent.TimeUnit;

import static sila_java.library.core.mapping.FeatureGenerator.generateFeature;

/**
 * Static Utilities for SiLA Server Loading
 */
@Slf4j
@NoArgsConstructor(access = AccessLevel.PRIVATE)
public abstract class ServerLoading {
    private final static long MAX_SERVICE_TIMEOUT = 10; // [s]

    /**
     * Load SiLA Server from SiLA Service
     *
     * @param server Server to provide SiLA Server information
     * @param managedChannel Connection to call the SiLA Server
     */
    public static void loadServer(@NonNull final Server server, @NonNull final ManagedChannel managedChannel) {
        // Call SiLAService
        final SiLAServiceGrpc.SiLAServiceBlockingStub siLAService = SiLAServiceGrpc.newBlockingStub(managedChannel);
        try {
            // Load Server Info
            final UUID serverUUID = getServerId(siLAService);
            log.info("Got serverUUID: {}", serverUUID);
            final String serverName = getServerName(siLAService);
            final String serverType = getServerType(siLAService);
            final String serverDescription = getServerDescription(siLAService, serverName);
            final String serverVendorURL = getServerVendor(siLAService);
            final String serverVersion = getServerVersion(siLAService);

            server.setConfiguration(new ServerConfiguration(serverName, serverUUID));
            server.setInformation(new ServerInformation(serverType, serverDescription, serverVendorURL, serverVersion));
            log.info("{} Saved Server Information", serverName);

            // Get FeatureList
            final SiLAServiceOuterClass.Get_ImplementedFeatures_Responses featureList =
                    getFeatureList(siLAService, serverName);

            // Compile single Features
            loadFeatures(siLAService, server, featureList);

            // Set Status to Online
            server.setStatus(Server.Status.ONLINE);
        } catch (StatusRuntimeException e) {
            final String errorMessage = String.format(
                    "SiLA Service doesn't return because %s. Timeout was set to %d s",
                    e.getMessage(),
                    MAX_SERVICE_TIMEOUT
            );
            log.warn(errorMessage);
            server.getErrorMessages().add(errorMessage);
        }
    }

    /**
     * Get Unique Identifier from SiLA Server
     * @param siLAService SiLA Service Blocking Stub
     * @return Unique Identifier
     */
    public static UUID getServerId(@NonNull final SiLAServiceGrpc.SiLAServiceBlockingStub siLAService) {
        final String serverUUIDStr = siLAService
                .withDeadlineAfter(MAX_SERVICE_TIMEOUT, TimeUnit.SECONDS)
                .getServerUUID(SiLAServiceOuterClass.Get_ServerUUID_Parameters.newBuilder().build())
                .getServerUUID()
                .getValue();
        return UUID.fromString(serverUUIDStr);
    }

    private static SiLAServiceOuterClass.Get_ImplementedFeatures_Responses getFeatureList(
            @NonNull final SiLAServiceGrpc.SiLAServiceBlockingStub siLAService,
            @NonNull final String serverName
    ) {
        final long start = System.currentTimeMillis();
        final SiLAServiceOuterClass.Get_ImplementedFeatures_Responses featureList = siLAService
                .withDeadlineAfter(MAX_SERVICE_TIMEOUT, TimeUnit.SECONDS)
                .getImplementedFeatures(
                        SiLAServiceOuterClass.Get_ImplementedFeatures_Parameters
                                .newBuilder()
                                .build()
                );
        log.info("{} Got List of Features in {} ms", serverName, (System.currentTimeMillis() - start));
        return featureList;
    }

    private static String getServerVersion(@NonNull final SiLAServiceGrpc.SiLAServiceBlockingStub siLAService) {
        final String serverVersion = siLAService
                .withDeadlineAfter(MAX_SERVICE_TIMEOUT, TimeUnit.SECONDS)
                .getServerVersion(SiLAServiceOuterClass.Get_ServerVersion_Parameters.newBuilder().build())
                .getServerVersion()
                .getValue();
        log.info("Got serverVersion: {}", serverVersion);
        return serverVersion;
    }

    private static String getServerVendor(@NonNull final SiLAServiceGrpc.SiLAServiceBlockingStub siLAService) {
        final String serverVendorURL = siLAService
                .withDeadlineAfter(MAX_SERVICE_TIMEOUT, TimeUnit.SECONDS)
                .getServerVendorURL(SiLAServiceOuterClass.Get_ServerVendorURL_Parameters.newBuilder().build())
                .getServerVendorURL()
                .getURL()
                .getValue();
        log.info("Got serverVendorURL: {}", serverVendorURL);
        return serverVendorURL;
    }

    private static String getServerDescription(
            @NonNull final SiLAServiceGrpc.SiLAServiceBlockingStub siLAService,
            @NonNull final String serverName
    ) {
        final String serverDescription = siLAService
                .withDeadlineAfter(MAX_SERVICE_TIMEOUT, TimeUnit.SECONDS)
                .getServerDescription(SiLAServiceOuterClass.Get_ServerDescription_Parameters.newBuilder().build())
                .getServerDescription()
                .getValue();
        log.info("{} Got serverDescription: {}", serverName, serverDescription);
        return serverDescription;
    }

    private static String getServerType(@NonNull final SiLAServiceGrpc.SiLAServiceBlockingStub siLAService) {
        final String serverType = siLAService
                .withDeadlineAfter(MAX_SERVICE_TIMEOUT, TimeUnit.SECONDS)
                .getServerType(SiLAServiceOuterClass.Get_ServerType_Parameters.newBuilder().build())
                .getServerType()
                .getValue();
        log.info("Got serverType: {}", serverType);
        return serverType;
    }

    private static String getServerName(@NonNull final SiLAServiceGrpc.SiLAServiceBlockingStub siLAService) {
        final String serverName = siLAService
                .withDeadlineAfter(MAX_SERVICE_TIMEOUT, TimeUnit.SECONDS)
                .getServerName(SiLAServiceOuterClass.Get_ServerName_Parameters.newBuilder().build())
                .getServerName()
                .getValue();
        log.info("Got serverName: {}", serverName);
        return serverName;
    }

    /**
     * Loading the Features into the SiLA Server Model
     *
     * @param siLAService Stub to retrieve the feature definitions
     * @param server SiLA Server model
     * @param featureList Feature List retrieved from SiLA Server
     */
    private static void loadFeatures(
            @NonNull final SiLAServiceGrpc.SiLAServiceBlockingStub siLAService,
            @NonNull final Server server,
            @NonNull final SiLAServiceOuterClass.Get_ImplementedFeatures_Responses featureList
    ) {
        featureList.getImplementedFeaturesList().forEach(featureIdentifier -> {
            final SiLAServiceOuterClass.GetFeatureDefinition_Parameters par =
                    SiLAServiceOuterClass.GetFeatureDefinition_Parameters.newBuilder()
                            .setQualifiedFeatureIdentifier(featureIdentifier)
                            .build();

            final SiLAServiceOuterClass.GetFeatureDefinition_Responses featureDefinition = siLAService
                    .withDeadlineAfter(MAX_SERVICE_TIMEOUT, TimeUnit.SECONDS)
                    .getFeatureDefinition(par);

            final String rawFeatureDefinition = featureDefinition
                    .getFeatureDefinition()
                    .getFeatureDefinition()
                    .getValue();
            try {
                // Deserialize into feature
                final Feature featurePojo = generateFeature(new CharSequenceReader(rawFeatureDefinition));

                log.info("Feature {}=", featureIdentifier);
                log.info("{} ", featurePojo);

                server.getFeatures().add(featurePojo);
            } catch (Exception e) {
                final String errorMessage = String.format(
                        "Parsing of Feature %s failed. Reason=%s",
                        featureIdentifier,
                        e.getMessage()
                );

                log.warn(errorMessage);
                server.getErrorMessages().add(errorMessage);
            }
        });
    }
}
