/*
 * Decompiled with CFR 0.152.
 */
package sila_java.library.manager;

import com.google.protobuf.Descriptors;
import io.grpc.ManagedChannel;
import java.security.KeyException;
import java.time.Duration;
import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.UUID;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.TimeoutException;
import java.util.stream.Collectors;
import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import lombok.NonNull;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import sila_java.library.core.asynchronous.MethodPoller;
import sila_java.library.core.discovery.ServerListener;
import sila_java.library.core.discovery.SiLAServerDiscovery;
import sila_java.library.core.mapping.MalformedSiLAFeature;
import sila_java.library.core.mapping.ProtoMapper;
import sila_java.library.core.models.Feature;
import sila_java.library.manager.executor.ServerCallExecutor;
import sila_java.library.manager.models.Server;
import sila_java.library.manager.models.SiLACall;
import sila_java.library.manager.server_management.DiscoveryListener;
import sila_java.library.manager.server_management.ServerHeartbeat;
import sila_java.library.manager.server_management.ServerLoading;
import sila_java.library.manager.server_management.ServerUtilities;
import sila_java.library.manager.server_management.SiLAConnection;

public class SiLAManager {
    private static final Logger log = LoggerFactory.getLogger(SiLAManager.class);
    private static final int HEARTBEAT = 1000;
    private static SiLAManager Instance;
    private final Map<UUID, Server> silaServers = new ConcurrentHashMap<UUID, Server>();
    private final Map<UUID, SiLAConnection> silaConnections = new ConcurrentHashMap<UUID, SiLAConnection>();
    private final List<SiLAServerListener> siLAServerListenerList = new CopyOnWriteArrayList<SiLAServerListener>();
    private final DiscoveryListener discoveryListener = new DiscoveryListener(this);
    private final ServerHeartbeat serverHeartbeat;
    private boolean discovery = false;

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public static SiLAManager getInstance() {
        Class<SiLAManager> clazz = SiLAManager.class;
        synchronized (SiLAManager.class) {
            if (Instance == null) {
                Instance = new SiLAManager();
            }
            // ** MonitorExit[var0] (shouldn't be in output)
            return Instance;
        }
    }

    private SiLAManager() {
        SiLAServerDiscovery.getInstance().addListener((ServerListener)this.discoveryListener);
        this.serverHeartbeat = new ServerHeartbeat(this, this.discoveryListener, 1000);
    }

    public void initialize(@NonNull Map<UUID, Server> silaServers) {
        if (silaServers == null) {
            throw new NullPointerException("silaServers");
        }
        if (!this.silaServers.isEmpty() || !this.silaConnections.isEmpty()) {
            throw new IllegalStateException("SiLA Manager can only be initialised in an empty state");
        }
        log.info("Initializing SiLA Manager with " + silaServers.size() + " servers");
        this.silaServers.putAll(silaServers);
        this.silaServers.forEach((uuid, server) -> {
            ManagedChannel managedChannel = ServerUtilities.buildNewChannel(server);
            this.silaConnections.put((UUID)uuid, this.createSiLAConnection(managedChannel, (Server)server));
        });
    }

    public void startDiscovery() {
        if (!this.discovery) {
            log.debug("SiLA Manager started");
            this.discovery = true;
            SiLAServerDiscovery.getInstance().start();
        } else {
            log.info("An attempt was made to start the SiLA Manager discovery when it was already running");
        }
    }

    public void stopDiscovery() {
        if (this.discovery) {
            log.debug("SiLA Manager stopped");
            this.discovery = false;
            SiLAServerDiscovery.getInstance().stop();
        } else {
            log.info("An attempt was made to stop the SiLA Manager discovery when it was already stopped");
        }
    }

    public void startHeartbeat() {
        this.serverHeartbeat.start();
    }

    public void stopHeartbeat() {
        this.serverHeartbeat.stop();
    }

    public void addServer(@NonNull String host, int port) {
        if (host == null) {
            throw new NullPointerException("host");
        }
        this.addServer(host, port, null);
    }

    public synchronized void removeServer(@NonNull UUID id) {
        if (id == null) {
            throw new NullPointerException("id");
        }
        if (!this.silaServers.containsKey(id)) {
            log.info("Server with id: " + id + " already removed from cache.");
            return;
        }
        UUID discoveryId = this.silaServers.get(id).getDiscoveryId();
        this.silaServers.remove(id);
        if (this.silaConnections.containsKey(id)) {
            this.silaConnections.get(id).close();
            this.silaConnections.remove(id);
        }
        if (discoveryId != null) {
            this.discoveryListener.removeFromCache(discoveryId);
        }
        log.info("[removeServer] removed serverId={}", (Object)id);
        this.siLAServerListenerList.forEach(listener -> listener.onSiLAServerRemoved(id));
    }

    public void addServer(@NonNull String host, int port, @Nullable UUID discoveryId) {
        if (host == null) {
            throw new NullPointerException("host");
        }
        Server server = new Server(discoveryId);
        server.setJoined(new Date());
        server.setHost(host);
        server.setPort(port);
        server.setStatus(Server.Status.INVALID);
        this.parseServer(server);
    }

    private synchronized void parseServer(@NonNull Server server) {
        UUID serverKeyUUID;
        if (server == null) {
            throw new NullPointerException("server");
        }
        ManagedChannel managedChannel = ServerUtilities.buildNewChannel(server);
        ServerLoading.loadServer(server, managedChannel);
        if (server.getStatus().equals((Object)Server.Status.INVALID)) {
            log.warn("Server saved in cache, but Invalid and with random UUID Key.");
            serverKeyUUID = UUID.randomUUID();
        } else {
            serverKeyUUID = server.getConfiguration().getUuid();
            log.info("[addServer] Resolved SiLA Server serverName={} on {}:{}", new Object[]{server.getConfiguration().getName(), server.getHost(), server.getPort()});
        }
        if (this.silaServers.containsKey(serverKeyUUID)) {
            log.warn("Server with id: " + serverKeyUUID + " was already in cache - overwritng.");
        }
        this.silaServers.put(serverKeyUUID, server);
        this.silaConnections.put(serverKeyUUID, this.createSiLAConnection(managedChannel, server));
        this.siLAServerListenerList.forEach(listener -> listener.onSiLAServerAdded(serverKeyUUID, server));
    }

    private SiLAConnection createSiLAConnection(@NonNull ManagedChannel managedChannel, @NonNull Server server) {
        if (managedChannel == null) {
            throw new NullPointerException("managedChannel");
        }
        if (server == null) {
            throw new NullPointerException("server");
        }
        SiLAConnection siLAConnection = new SiLAConnection(managedChannel);
        ProtoMapper protoMapper = new ProtoMapper();
        for (Feature feature : server.getFeatures()) {
            try {
                String featureIdentifier = SiLAManager.getFeatureId(feature.getIdentifier());
                Descriptors.FileDescriptor protoFile = protoMapper.getProto(feature);
                siLAConnection.addFeatureService(featureIdentifier, (Descriptors.ServiceDescriptor)protoFile.getServices().get(0));
            }
            catch (Descriptors.DescriptorValidationException | MalformedSiLAFeature e) {
                server.setStatus(Server.Status.INVALID);
                break;
            }
        }
        return siLAConnection;
    }

    public void setServerStatus(@NonNull UUID id, @NonNull Server.Status status) {
        if (id == null) {
            throw new NullPointerException("id");
        }
        if (status == null) {
            throw new NullPointerException("status");
        }
        Server server = this.silaServers.get(id);
        if (server == null) {
            log.warn("Server with id " + id.toString() + "doesnt exit.");
            return;
        }
        if (!server.getStatus().equals((Object)status)) {
            server.setStatus(status);
            log.info("[changeServerStatus] change {} status to {}", (Object)id.toString(), (Object)status.toString());
            this.siLAServerListenerList.forEach(listener -> listener.onServerChange(id));
        }
    }

    public Map<UUID, Server> getSiLAServers() {
        return this.silaServers;
    }

    public Map<UUID, SiLAConnection> getSilaConnections() {
        return this.silaConnections;
    }

    public List<Server> getSiLAServersByName(@NonNull String serverName) {
        if (serverName == null) {
            throw new NullPointerException("serverName");
        }
        return this.silaServers.values().stream().filter(server -> server.getConfiguration().getName().equals(serverName)).collect(Collectors.toList());
    }

    public void addSiLAServerListener(@Nonnull SiLAServerListener siLAServerListener) {
        this.siLAServerListenerList.add(siLAServerListener);
    }

    public void removeSiLAServerListener(@Nonnull SiLAServerListener siLAServerListener) {
        this.siLAServerListenerList.remove(siLAServerListener);
    }

    public ServerCallExecutor newCallExecutor(@NonNull SiLACall siLACall) throws KeyException {
        if (siLACall == null) {
            throw new NullPointerException("siLACall");
        }
        SiLAConnection connection = this.silaConnections.get(siLACall.getServerId());
        return new ServerCallExecutor(connection, siLACall);
    }

    public void blockUntilServerFound(@NonNull String serverName, int timeOut) throws TimeoutException {
        if (serverName == null) {
            throw new NullPointerException("serverName");
        }
        try {
            MethodPoller.await().atMost(Duration.ofMillis(timeOut)).until(() -> this.getSiLAServersByName(serverName).stream().filter(server -> server.getStatus().equals((Object)Server.Status.ONLINE)).anyMatch(server -> server.getConfiguration().getName().equals(serverName))).execute();
        }
        catch (ExecutionException e) {
            throw new RuntimeException(e.getCause());
        }
        catch (TimeoutException e) {
            throw new TimeoutException("Can not find SiLA Server " + serverName + " in " + timeOut + " ms");
        }
        log.info("Found SiLA Server " + serverName);
    }

    private static String getFeatureId(@NonNull String featureIdentifier) {
        if (featureIdentifier == null) {
            throw new NullPointerException("featureIdentifier");
        }
        int lastIndexOf = featureIdentifier.lastIndexOf(47);
        if (lastIndexOf > 0 && lastIndexOf != featureIdentifier.length() - 1) {
            return featureIdentifier.substring(lastIndexOf + 1);
        }
        return featureIdentifier;
    }

    public static interface SiLAServerListener {
        public void onSiLAServerAdded(UUID var1, Server var2);

        public void onSiLAServerRemoved(UUID var1);

        public void onServerChange(UUID var1);
    }
}

