package sila_java.library.server_base;

import io.grpc.*;
import lombok.NonNull;
import lombok.extern.slf4j.Slf4j;
import sila_java.library.core.discovery.SiLAServerRegistration;
import sila_java.library.server_base.config.IServerConfigWrapper;
import sila_java.library.server_base.config.NonPersistentServerConfigWrapper;
import sila_java.library.server_base.config.PersistentServerConfigWrapper;
import sila_java.library.server_base.identification.ServerInformation;

import java.io.Closeable;
import java.io.IOException;
import java.nio.file.Path;
import java.util.Arrays;
import java.util.Map;
import java.util.concurrent.TimeUnit;

@Slf4j
public class SiLAServerBase implements Closeable {
    private final static int SHUTDOWN_TIMEOUT = 20; // [s]
    private final SiLAServerRegistration serverRegistration = new SiLAServerRegistration();
    private final Server server;

    /**
     * Create and start a Base Server with a non-persistent configuration to expose the SiLA Features
     *
     * @param serverInformation     Meta server information defined by the server implementer
     * @param featureDefinitions    Map with FeatureIdentifier as Keys and and the
     *                              Filepath or Content of the FDL as Values
     * @param port                  Port on which the server runs.
     * @param interfaceName         Name of network interface to use discovery
     * @param bindableServices      Variadic inputs of Service Implementations to bind
     *
     * @implNote Instance can only start one server at a time.
     */
    public static SiLAServerBase withoutConfig(
            @NonNull final ServerInformation serverInformation,
            @NonNull final Map<String, String> featureDefinitions,
            final int port,
            @NonNull final String interfaceName,
            final BindableService... bindableServices
    ) throws IOException {
        log.debug("Server config is non-persistent");
        return new SiLAServerBase(
                new NonPersistentServerConfigWrapper(serverInformation.getType()),
                serverInformation,
                featureDefinitions,
                port,
                interfaceName,
                bindableServices
        );
    }

    /**
     * Create and start a Base Server with a persistent configuration file to expose the SiLA Features
     *
     * @param configurationFile     The file persisting the server name and UUID data for that server instance
     * @param serverInformation     Meta server information defined by the server implementer
     * @param featureDefinitions    Map with FeatureIdentifier as Keys and and the
     *                              Filepath or Content of the FDL as Values
     * @param port                  Port on which the server runs.
     * @param interfaceName         Name of network interface to use discovery
     * @param bindableServices      Variadic inputs of Service Implementations to bind
     *
     * @implNote Instance can only start one server at a time.
     */
    public static SiLAServerBase withConfig(
            @NonNull final Path configurationFile,
            @NonNull final ServerInformation serverInformation,
            @NonNull final Map<String, String> featureDefinitions,
            final int port,
            @NonNull final String interfaceName,
            BindableService... bindableServices
    ) throws IOException {
        log.debug("Server config is persistent");
        return new SiLAServerBase(
                new PersistentServerConfigWrapper(configurationFile, serverInformation.getType()),
                serverInformation,
                featureDefinitions,
                port,
                interfaceName,
                bindableServices
        );
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void close() {
        log.info("[stop] stopping server...");
        this.serverRegistration.unregisterServer();

        // Stop server
        if (this.server != null && !this.server.isTerminated() && !this.server.isShutdown()) {
            try {
                log.info("[stop] stopping the server ...");
                this.server.shutdownNow().awaitTermination(SHUTDOWN_TIMEOUT, TimeUnit.SECONDS);
                log.info("[stop] the server was stopped");
            } catch (InterruptedException e) {
                log.warn("[stop] could not shutdown the server within {} seconds", SHUTDOWN_TIMEOUT);
            }
            log.info("[stop] stopped");
        } else {
            log.info("[stop] server already stopped");
        }
    }

    /**
     * Create and start a Base Server to expose the SiLA Features
     *
     * @param serverConfig          The configuration container
     *                              If null, there server config will not be persistent
     * @param serverInformation     Meta server information defined by the server implementer
     * @param featureDefinitions    Map with FeatureIdentifier as Keys and and the
     *                              Filepath or Content of the FDL as Values
     * @param port                  Port on which the server runs.
     * @param interfaceName         Name of network interface to use discovery
     * @param bindableServices      Variadic inputs of Service Implementations to bind
     *
     * @implNote Instance can only withoutConfig one server at a time.
     */
    private SiLAServerBase(
            @NonNull final IServerConfigWrapper serverConfig,
            @NonNull final ServerInformation serverInformation,
            @NonNull final Map<String, String> featureDefinitions,
            final int port,
            @NonNull final String interfaceName,
            BindableService... bindableServices
    ) throws IOException {
        final SiLAServiceServer siLAServiceServer = new SiLAServiceServer(
                serverConfig,
                serverInformation,
                featureDefinitions
        );
        log.info("[withoutConfig] Server registered on SiLAService with features={}", featureDefinitions.keySet());

        final ServerBuilder serverBuilder = ServerBuilder.forPort(port).addService(siLAServiceServer.getService());

        Arrays.stream(bindableServices).forEach(serverBuilder::addService);

        this.server = serverBuilder.build().start();

        log.info("[withoutConfig] Server started on port={}", port);
        this.serverRegistration.registerServer(serverConfig.getCacheConfig().getUuid(), interfaceName, port);
        log.info("[withoutConfig] Server registered with discovery.");

        Runtime.getRuntime().addShutdownHook(new Thread(this::close));
    }
}
