package sila_java.library.server_base;

import com.google.gson.JsonObject;
import io.grpc.Status;
import io.grpc.StatusRuntimeException;

import javax.annotation.Nonnull;
import javax.xml.transform.stream.StreamSource;
import java.io.IOException;
import java.io.StringReader;
import java.util.HashMap;
import java.util.Map;

import io.grpc.stub.StreamObserver;
import lombok.NonNull;
import lombok.extern.slf4j.Slf4j;

import lombok.val;
import sila2.org.silastandard.SiLAFramework;
import sila2.org.silastandard.core.silaservice.v1.SiLAServiceGrpc;
import sila2.org.silastandard.core.silaservice.v1.SiLAServiceOuterClass;
import sila_java.library.core.utils.SiLAErrors;
import sila_java.library.server_base.config.IServerConfigWrapper;
import sila_java.library.server_base.identification.ServerInformation;

import static sila_java.library.core.utils.Utils.validateFeatureXML;

/**
 * SiLAService Implementation used by other SiLA servers
 */
@Slf4j
public class SiLAServiceServer {
    private final Map<String, String> featureDefinitions = new HashMap<>();

    private final IServerConfigWrapper serverConfigurationContainer;
    private final ServerInformation serverInformation;

    public SiLAServiceServer(
            @NonNull final IServerConfigWrapper serverConfigurationContainer,
            @Nonnull final ServerInformation serverInformation,
            @Nonnull final Map<String, String> featureDefinitions) {
        this.serverConfigurationContainer = serverConfigurationContainer;
        this.serverInformation = serverInformation;

        log.info(
                "[SiLAService] create service for: {}\nfeatureDefinitions={} ...",
                serverInformation,
                featureDefinitions.keySet()
        );

        // Get serialised feature definition
        for (final String key : featureDefinitions.keySet()) {
            try {
                final String featureDefinition = featureDefinitions.get(key);

                validateFeatureXML(new StreamSource(new StringReader(featureDefinition)));
                this.featureDefinitions.put(key, featureDefinition);
                log.info(
                        "[registerServer] type={} feature={} is read from XML string.",
                        serverInformation.getType(),
                        key
                );
            } catch (IOException e) {
                throw new IllegalArgumentException(
                        "Error registering feature definition for server = " + serverInformation.getType() +
                                "& feature = '" + key + "' is not valid", e);
            }
        }
        log.info(
                "[registerServer] server type = {} with features successfully registered!",
                serverInformation.getType()
        );
    }


    /**
     * Gets gRPC Service to serve
     *
     * @return bindable service
     */
    public io.grpc.BindableService getService() {
        return new ServiceImpl();
    }

    /**
     * gRPC Service Implementation
     */
    private class ServiceImpl extends SiLAServiceGrpc.SiLAServiceImplBase {
        @Override
        public void getFeatureDefinition(
                @NonNull final SiLAServiceOuterClass.GetFeatureDefinition_Parameters req,
                @NonNull final StreamObserver<SiLAServiceOuterClass.GetFeatureDefinition_Responses> responseObserver
        ) {
            final String featureIdentifier = req
                    .getQualifiedFeatureIdentifier()
                    .getFeatureIdentifier()
                    .getValue();

            final String serverName = serverConfigurationContainer.getCacheConfig().getName();

            log.info(
                    "[{}][getFeatureDefinition] feature={} entered.",
                    serverName,
                    featureIdentifier
            );

            final String featureDefinitionContent =
                    SiLAServiceServer.this.featureDefinitions.get(featureIdentifier);

            if (featureDefinitionContent == null) {
                JsonObject validationError = new JsonObject();
                validationError.addProperty("Parameter", "FeatureIdentifier");
                validationError.addProperty("Message", "Feature definition for " +
                        featureIdentifier + " doesn't exist, " +
                        " existing Features: " + featureDefinitions.keySet());

                responseObserver.onError(new StatusRuntimeException(Status.INVALID_ARGUMENT
                        .withDescription(validationError.toString())));
                return;
            }
            responseObserver.onNext(
                    SiLAServiceOuterClass.GetFeatureDefinition_Responses.newBuilder()
                            .setFeatureDefinition(
                                    SiLAServiceOuterClass.DataType_FeatureDefinition
                                            .newBuilder()
                                            .setFeatureDefinition(
                                                    SiLAFramework.String.newBuilder()
                                                            .setValue(featureDefinitionContent)
                                                            .build()
                                                    )
                                            .build()
                            )
                            .build()
            );
            responseObserver.onCompleted();
            log.info(
                    "[{}][getFeatureDefinition] feature={} returned={}.",
                    serverName,
                    featureIdentifier,
                    featureDefinitionContent
            );
        }

        @Override
        public void setServerName(
                SiLAServiceOuterClass.SetServerName_Parameters req,
                StreamObserver<SiLAServiceOuterClass.SetServerName_Responses> responseObserver
        ) {
            val serverName = req.getServerName().getValue();
            log.info("setServerName being called, Server Name: {}", serverName);

            try {
                serverConfigurationContainer.setConfig(
                        serverConfigurationContainer.getCacheConfig().withName(serverName)
                );
                responseObserver.onNext(SiLAServiceOuterClass.SetServerName_Responses.newBuilder().build());
                responseObserver.onCompleted();
            } catch (IOException e) {
                val error = SiLAErrors.generateGenericExecutionError(e);
                responseObserver.onError(error);
            }
        }


        @Override
        public void getServerName(
                SiLAServiceOuterClass.Get_ServerName_Parameters req,
                StreamObserver<SiLAServiceOuterClass.Get_ServerName_Responses> responseObserver
        ) {
            log.info("getServerName being called");

            responseObserver.onNext(
                    SiLAServiceOuterClass.Get_ServerName_Responses.newBuilder()
                            .setServerName(
                                    SiLAFramework.String.newBuilder()
                                            .setValue(serverConfigurationContainer.getCacheConfig().getName())
                                            .build()

                            )
                            .build()
            );
            responseObserver.onCompleted();
        }

        @Override
        public void getServerType(
                SiLAServiceOuterClass.Get_ServerType_Parameters req,
                StreamObserver<SiLAServiceOuterClass.Get_ServerType_Responses> responseObserver
        ) {
            log.info("getServerType being called");

            responseObserver.onNext(
                    SiLAServiceOuterClass.Get_ServerType_Responses.newBuilder()
                            .setServerType(
                                    SiLAFramework.String.newBuilder().setValue(serverInformation.getType()).build()
                            )
                            .build()
            );
            responseObserver.onCompleted();
        }

        @Override
        public void getServerUUID(
                SiLAServiceOuterClass.Get_ServerUUID_Parameters req,
                StreamObserver<SiLAServiceOuterClass.Get_ServerUUID_Responses> responseObserver
        ) {
            responseObserver.onNext(
                    SiLAServiceOuterClass.Get_ServerUUID_Responses.newBuilder()
                            .setServerUUID(
                                    SiLAFramework.String.newBuilder()
                                            .setValue(serverConfigurationContainer.getCacheConfig().getUuid().toString())
                                            .build()
                            )
                            .build()
            );
            responseObserver.onCompleted();
        }

        @Override
        public void getServerDescription(
                @NonNull final SiLAServiceOuterClass.Get_ServerDescription_Parameters req,
                @NonNull final StreamObserver<SiLAServiceOuterClass.Get_ServerDescription_Responses> responseObserver
        ) {
            log.info("getServerDescription being called");

            responseObserver.onNext(
                    SiLAServiceOuterClass.Get_ServerDescription_Responses.newBuilder()
                        .setServerDescription(
                                SiLAFramework.String.newBuilder()
                                        .setValue(serverInformation.getDescription())
                                        .build()
                        )
                        .build()
            );
            responseObserver.onCompleted();
        }

        @Override
        public void getServerVersion(
                @NonNull final SiLAServiceOuterClass.Get_ServerVersion_Parameters req,
                @NonNull final StreamObserver<SiLAServiceOuterClass.Get_ServerVersion_Responses> responseObserver
        ) {
            log.info("getServerVersion being called");

            responseObserver.onNext(
                    SiLAServiceOuterClass.Get_ServerVersion_Responses.newBuilder()
                            .setServerVersion(
                                    SiLAFramework.String.newBuilder().setValue(serverInformation.getVersion()).build()
                            )
                            .build()
            );
            responseObserver.onCompleted();
        }

        @Override
        public void getServerVendorURL(
                @NonNull final SiLAServiceOuterClass.Get_ServerVendorURL_Parameters req,
                @NonNull final StreamObserver<SiLAServiceOuterClass.Get_ServerVendorURL_Responses> responseObserver
        ) {
            log.info("getServerVendorName being called");

            responseObserver.onNext(SiLAServiceOuterClass.Get_ServerVendorURL_Responses.newBuilder()
                    .setServerVendorURL(
                            SiLAServiceOuterClass.DataType_URL.newBuilder()
                                    .setURL(
                                            SiLAFramework.String.newBuilder()
                                                    .setValue(serverInformation.getVendorURL())
                                                    .build()
                                    )
                                    .build()
                    )
                    .build());
            responseObserver.onCompleted();
        }

        @Override
        public void getImplementedFeatures(
                @NonNull final SiLAServiceOuterClass.Get_ImplementedFeatures_Parameters req,
                @NonNull final StreamObserver<SiLAServiceOuterClass.Get_ImplementedFeatures_Responses> responseObserver
        ) {
            log.info("getImplementedFeatures being called");

            final SiLAServiceOuterClass.Get_ImplementedFeatures_Responses.Builder idBuilder =
                    SiLAServiceOuterClass.Get_ImplementedFeatures_Responses.newBuilder();

            featureDefinitions.keySet().forEach(featureId -> idBuilder.addImplementedFeatures(
                SiLAServiceOuterClass.DataType_FeatureIdentifier.newBuilder()
                        .setFeatureIdentifier(SiLAFramework.String.newBuilder().setValue(featureId).build())
                        .build())
            );

            responseObserver.onNext(idBuilder.build());
            responseObserver.onCompleted();
        }
    }
}