package sila_java.library.server_base.utils;

import lombok.Getter;
import lombok.NonNull;
import net.sourceforge.argparse4j.ArgumentParsers;
import net.sourceforge.argparse4j.impl.Arguments;
import net.sourceforge.argparse4j.inf.ArgumentParser;
import net.sourceforge.argparse4j.inf.Namespace;
import sila_java.library.core.communication.utils.ListNetworkInterfaces;

import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.Optional;

public class ArgumentHelper {
    private final ArgumentParser parser;
    @Getter private final String[] args;
    @Getter private final Namespace ns;

    /**
     * Parse the main arguments
     * If the list network flag is specified, a list of internet interface will be displayed and the program will exit
     * @param args The main arguments
     * @param serverName The server name
     */
    public ArgumentHelper(@NonNull final String[] args,
                          @NonNull final String serverName) {
        this(args, serverName, null);
    }

    /**
     * Parse the main arguments
     * If the list network flag is specified, a list of internet interface will be displayed and the program will exit
     * @param args The main arguments
     * @param serverName The server name
     * @param baseParser If you want to extend the default parser with other flags
     */
    public ArgumentHelper(@NonNull final String[] args,
                          @NonNull final String serverName,
                          final ArgumentParser baseParser) {
        this.args = args;
        // Argument Parser
        if (baseParser == null) {
            parser = ArgumentParsers.newFor(serverName).build()
                    .defaultHelp(true)
                    .description("SiLA Server.");
        } else {
            parser = baseParser;
        }
        // Port Argument (optional)
        parser.addArgument("-p", "--port")
                .type(Integer.class)
                .help("Specify port to use.");
        // Network Interface Argument
        parser.addArgument("-n", "--networkInterface")
                .type(String.class)
                .setDefault("local")
                .help("Specify network interface. Check ifconfig (LNX & MAC)"
                        + " and for windows, ask us for a tiny java app @ www.unitelabs.ch .");
        // Server config file holding UUID and ServerName
        parser.addArgument("-c", "--configFile")
                .type(String.class)
                .help("Specify the file name to use to read/store server information");
        // Only list System Information
        parser.addArgument("-l", "--listNetworks")
                .type(Arguments.booleanType("yes", "no"))
                .setDefault(false)
                .help("List names of network interfaces found.");
        this.ns = parser.parseArgsOrFail(args);
        if (ns.getBoolean("listNetworks")) {
            ListNetworkInterfaces.display();
            System.exit(0);
        }
    }

    public Integer getPort() {
        return (ns.getInt("port"));
    }

    public String getInterface() {
        return (ns.getString("networkInterface"));
    }

    public Optional<Path> getConfigFile() {
        final String configFile = ns.getString("configFile");
        if (configFile == null)
            return Optional.empty();
        return Optional.of(Paths.get(configFile));
    }
}
