/*
 * Decompiled with CFR 0.152.
 */
package sila_java.library.server_base;

import com.google.common.collect.ArrayListMultimap;
import com.google.common.collect.Multimap;
import io.grpc.BindableService;
import io.grpc.InsecureServerCredentials;
import io.grpc.Server;
import io.grpc.ServerBuilder;
import io.grpc.ServerCredentials;
import io.grpc.ServerInterceptor;
import io.grpc.ServerServiceDefinition;
import io.grpc.TlsServerCredentials;
import io.grpc.netty.NettyServerBuilder;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.net.InetAddress;
import java.net.InetSocketAddress;
import java.net.SocketAddress;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.security.PrivateKey;
import java.security.cert.CertificateException;
import java.security.cert.X509Certificate;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.UUID;
import java.util.concurrent.TimeUnit;
import java.util.stream.Stream;
import javax.annotation.Nullable;
import lombok.NonNull;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import sila_java.library.core.discovery.SiLAServerRegistration;
import sila_java.library.core.encryption.EncryptionUtils;
import sila_java.library.core.encryption.SelfSignedCertificate;
import sila_java.library.core.models.Feature;
import sila_java.library.core.sila.mapping.feature.FeatureGenerator;
import sila_java.library.core.sila.utils.FullyQualifiedIdentifierUtils;
import sila_java.library.core.utils.SocketUtils;
import sila_java.library.server_base.binary_transfer.database.BinaryDatabase;
import sila_java.library.server_base.binary_transfer.database.BinaryDatabaseInjector;
import sila_java.library.server_base.binary_transfer.database.impl.H2BinaryDatabase;
import sila_java.library.server_base.binary_transfer.download.DownloadService;
import sila_java.library.server_base.binary_transfer.upload.UploadService;
import sila_java.library.server_base.config.IServerConfigWrapper;
import sila_java.library.server_base.config.NonPersistentServerConfigWrapper;
import sila_java.library.server_base.config.PersistentServerConfigWrapper;
import sila_java.library.server_base.config.ServerConfiguration;
import sila_java.library.server_base.identification.ServerInformation;
import sila_java.library.server_base.metadata.MetadataExtractingInterceptor;
import sila_java.library.server_base.standard_features.FeatureImplementation;
import sila_java.library.server_base.standard_features.SiLAServiceServer;
import sila_java.library.server_base.utils.TransmitThrowableInterceptor;

public class SiLAServer
implements AutoCloseable {
    private static final Logger log = LoggerFactory.getLogger(SiLAServer.class);
    private static final int SHUTDOWN_TIMEOUT = 20;
    private static final String DEFAULT_NETWORK_INTERFACE = "local";
    private final SiLAServerRegistration serverRegistration;
    private final Server server;
    private final BinaryDatabase binaryDatabaseImpl;
    private final boolean ownsBinaryDatabase;
    private final Optional<DownloadService> downloadService;
    private final Optional<UploadService> uploadService;
    private final Thread shutdownHook = new Thread(this::close);
    private final Multimap<String, String> affectedByMetadata;

    /*
     * Enabled force condition propagation
     * Lifted jumps to return sites
     */
    private SiLAServer(@NonNull Builder builder) throws IOException {
        ServerCredentials serverCredentials;
        if (builder == null) {
            throw new NullPointerException("builder is marked non-null but is null");
        }
        IServerConfigWrapper serverConfigurationWrapper = builder.getNewServerConfigurationWrapper();
        this.affectedByMetadata = builder.affectedByMetadata;
        SiLAServiceServer siLAServiceServer = new SiLAServiceServer(serverConfigurationWrapper, builder.serverInformation, builder.featureDefinitions);
        InetAddress inetAddress = builder.host != null ? InetAddress.getByName(builder.host) : null;
        log.info("Server registered on SiLAService with features={}", builder.featureDefinitions.keySet());
        UUID serverUuid = serverConfigurationWrapper.getCacheConfig().getUuid();
        if (builder.discovery && builder.interfaceName == null && builder.host == null) {
            builder.interfaceName = DEFAULT_NETWORK_INTERFACE;
        }
        if (builder.discovery && builder.interfaceName != null && builder.host != null) {
            throw new RuntimeException("Discovery requires either a host or interface name but both were provided");
        }
        if (builder.privateKeyFile != null && builder.certificateFile != null) {
            File privateKeyFile = builder.privateKeyFile.toFile();
            File certificateFile = builder.certificateFile.toFile();
            boolean privateKeyFileExist = Files.exists(builder.privateKeyFile, new LinkOption[0]);
            boolean certificateFileExist = Files.exists(builder.certificateFile, new LinkOption[0]);
            if (!privateKeyFileExist && certificateFileExist) {
                throw new RuntimeException("Certificate was provided, but couldn't find Private Key");
            }
            if (!certificateFileExist && privateKeyFileExist) {
                throw new RuntimeException("Private Key was provided, but couldn't find Certificate");
            }
            if (privateKeyFileExist && certificateFileExist) {
                builder.privateKey = EncryptionUtils.readPrivateKey((File)privateKeyFile, (String)builder.certificatePassword);
                builder.certificate = EncryptionUtils.readCertificate((File)certificateFile);
            } else {
                privateKeyFile.getParentFile().mkdirs();
                privateKeyFile.createNewFile();
                certificateFile.getParentFile().mkdirs();
                certificateFile.createNewFile();
            }
        }
        if (builder.unsafeCommunication) {
            log.warn("Using plain-text communication forbidden by the SiLA 2 Standard !!");
        } else if (builder.certificate == null && builder.privateKey == null) {
            SelfSignedCertificate selfSignedCertificate;
            log.info("No certificate provided, creating and using self signed certificate");
            try {
                String serverIp = null;
                if (inetAddress != null) {
                    serverIp = inetAddress.getHostAddress();
                } else if (builder.interfaceName != null) {
                    try (SiLAServerRegistration tmpRegsitration = SiLAServerRegistration.newInstanceFromInterface((UUID)serverUuid, (String)builder.interfaceName, (int)builder.port, null);){
                        serverIp = tmpRegsitration.findInetAddress().getHostAddress();
                    }
                    catch (Exception e) {
                        log.warn("Failed to retrieve server ip for discovery!", (Throwable)e);
                    }
                }
                log.info("Creating self signed certificate for server uuid {} and ip {}", (Object)serverUuid, (Object)serverIp);
                selfSignedCertificate = SelfSignedCertificate.newBuilder().withServerUUID(serverUuid).withServerIP(serverIp).build();
            }
            catch (SelfSignedCertificate.CertificateGenerationException e) {
                throw new RuntimeException(e);
            }
            builder.certificate = selfSignedCertificate.getCertificate();
            builder.privateKey = selfSignedCertificate.getPrivateKey();
            if (builder.certificateFile != null) {
                EncryptionUtils.writeCertificateToFile((File)builder.certificateFile.toFile(), (X509Certificate)builder.certificate);
                log.info("Wrote certificate to {}", (Object)builder.certificateFile);
            }
            if (builder.privateKeyFile != null) {
                log.info("Wrote private key to {}", (Object)builder.privateKeyFile);
                EncryptionUtils.writePrivateKeyToFile((File)builder.privateKeyFile.toFile(), (PrivateKey)builder.privateKey);
            }
        }
        if (builder.certificate != null && builder.privateKey != null && !builder.unsafeCommunication) {
            try (InputStream certChainStream = EncryptionUtils.certificateToStream((X509Certificate)builder.certificate);
                 InputStream keyStream = EncryptionUtils.keyToStream((PrivateKey)builder.privateKey);){
                serverCredentials = TlsServerCredentials.create((InputStream)certChainStream, (InputStream)keyStream);
            }
            catch (CertificateException e) {
                throw new IOException(e);
            }
            log.info("Server will use safe encrypted communication.");
        } else {
            serverCredentials = InsecureServerCredentials.create();
            log.warn("Server will use deprecated unsafe plain-text communication.");
        }
        ServerBuilder serverBuilder = (builder.host == null ? NettyServerBuilder.forPort((int)builder.port, (ServerCredentials)serverCredentials) : NettyServerBuilder.forAddress((SocketAddress)new InetSocketAddress(builder.host, (int)builder.port), (ServerCredentials)serverCredentials)).addService(siLAServiceServer.getService());
        boolean bl = this.ownsBinaryDatabase = builder.customBinaryDatabaseProvider == null;
        if (builder.binaryTransferSupport) {
            log.info("Server will support binary transfer");
            this.binaryDatabaseImpl = builder.getNewBinaryDatabase(serverUuid);
            this.downloadService = Optional.of(new DownloadService(this.binaryDatabaseImpl));
            this.uploadService = Optional.of(new UploadService(this.binaryDatabaseImpl, siLAServiceServer.getFeatureDefinitions().values(), this.affectedByMetadata));
            serverBuilder.addService((BindableService)this.downloadService.get());
            serverBuilder.addService((BindableService)this.uploadService.get());
            if (builder.addBinaryDatabaseInjector) {
                serverBuilder.intercept((ServerInterceptor)new BinaryDatabaseInjector(this.binaryDatabaseImpl));
            }
        } else {
            this.binaryDatabaseImpl = null;
            log.warn("Server will not support binary transfer");
            this.downloadService = Optional.empty();
            this.uploadService = Optional.empty();
        }
        builder.bindableServices.forEach(arg_0 -> ((ServerBuilder)serverBuilder).addService(arg_0));
        builder.serverServices.forEach(arg_0 -> ((ServerBuilder)serverBuilder).addService(arg_0));
        if (builder.discovery) {
            String certificate;
            String string = certificate = builder.certificate != null && !builder.unsafeCommunication ? EncryptionUtils.writeCertificateToString((X509Certificate)builder.certificate) : null;
            if (builder.interfaceName != null) {
                this.serverRegistration = SiLAServerRegistration.newInstanceFromInterface((UUID)serverUuid, (String)builder.interfaceName, (int)builder.port, (String)certificate);
                log.info("Server registering with discovery on interface {}.", (Object)builder.interfaceName);
            } else {
                if (inetAddress == null) throw new RuntimeException("Discovery requires either a valid host or interface!");
                this.serverRegistration = SiLAServerRegistration.newInstanceFromHost((UUID)serverUuid, (InetAddress)inetAddress, (int)builder.port, (String)certificate);
                log.info("Server registering with discovery on host {}.", (Object)builder.host);
            }
        } else {
            this.serverRegistration = null;
            log.warn("Server starting without specifying a network interface, discovery is not enabled.");
        }
        builder.interceptors.forEach(serverInterceptor -> {
            serverBuilder.intercept(serverInterceptor);
            log.info("Added interceptor of type " + serverInterceptor.getClass().getTypeName());
        });
        this.server = serverBuilder.build().start();
        Runtime.getRuntime().addShutdownHook(this.shutdownHook);
        log.info("Server started on {}:{}", (Object)(builder.host == null ? "*" : builder.host), (Object)builder.port);
    }

    @Override
    public void close() {
        try {
            Runtime.getRuntime().removeShutdownHook(this.shutdownHook);
        }
        catch (IllegalStateException | SecurityException e) {
            log.debug("Failed to remove shutdown hook", (Throwable)e);
        }
        log.info("[stop] stopping server...");
        if (this.serverRegistration != null) {
            this.serverRegistration.close();
        }
        if (!this.server.isTerminated() && !this.server.isShutdown()) {
            try {
                log.info("[stop] stopping the server ...");
                this.server.shutdownNow().awaitTermination(20L, TimeUnit.SECONDS);
                log.info("[stop] the server was stopped");
            }
            catch (InterruptedException e) {
                log.warn("[stop] could not shutdown the server within {} seconds", (Object)20);
            }
            log.info("[stop] stopped");
        } else {
            log.info("[stop] server already stopped");
        }
        if (this.binaryDatabaseImpl != null && this.ownsBinaryDatabase) {
            try {
                this.binaryDatabaseImpl.close();
            }
            catch (Exception e) {
                log.warn("Error occurred while closing binary database {}", (Object)e.getMessage(), (Object)e);
            }
        }
    }

    public void blockUntilShutdown() {
        if (!this.server.isShutdown() && !this.server.isTerminated()) {
            this.server.awaitTermination();
        }
    }

    public BinaryDatabase getBinaryDatabaseImpl() {
        return this.binaryDatabaseImpl;
    }

    public boolean isOwnsBinaryDatabase() {
        return this.ownsBinaryDatabase;
    }

    public Optional<DownloadService> getDownloadService() {
        return this.downloadService;
    }

    public Optional<UploadService> getUploadService() {
        return this.uploadService;
    }

    public static class Builder {
        private static int[] defaultPortRange = new int[]{50052, 50308};
        private final ServerInformation serverInformation;
        private final Map<String, String> featureDefinitions = new HashMap<String, String>();
        private final List<BindableService> bindableServices = new ArrayList<BindableService>();
        private final List<ServerServiceDefinition> serverServices = new ArrayList<ServerServiceDefinition>();
        private String interfaceName = null;
        private boolean discovery = true;
        private Integer port = null;
        private String host = null;
        private ConfigWrapperProvider customServerConfigProvider = null;
        private Path persistentConfigFile = Paths.get("./server.cfg", new String[0]);
        private boolean persistedConfig = false;
        private String name = null;
        private UUID uuid = null;
        private ServerConfiguration defaultServerConfiguration;
        private boolean binaryTransferSupport = false;
        private BinaryDatabaseProvider customBinaryDatabaseProvider = null;
        private Path certificateFile = null;
        private String certificatePassword;
        private Path privateKeyFile = null;
        private X509Certificate certificate = null;
        private PrivateKey privateKey = null;
        private boolean unsafeCommunication = false;
        private final List<ServerInterceptor> interceptors = new ArrayList<ServerInterceptor>();
        private boolean addBinaryDatabaseInjector;
        private final Multimap<String, String> affectedByMetadata = ArrayListMultimap.create();

        public Map<String, String> getFeatureDefinitions() {
            return Collections.unmodifiableMap(this.featureDefinitions);
        }

        public Builder withDiscovery(@NonNull boolean isEnabled) {
            this.discovery = isEnabled;
            return this;
        }

        public Builder withoutDiscovery() {
            return this.withDiscovery(false);
        }

        public Builder withNetworkInterface(@NonNull String interfaceName) {
            if (interfaceName == null) {
                throw new NullPointerException("interfaceName is marked non-null but is null");
            }
            this.interfaceName = interfaceName;
            return this;
        }

        public Builder withPort(int port) {
            this.port = port;
            return this;
        }

        public Builder withHost(@Nullable String host) {
            this.host = host;
            return this;
        }

        public Builder withName(@Nullable String name) {
            this.name = name;
            return this;
        }

        public Builder withUUID(@Nullable UUID uuid) {
            this.uuid = uuid;
            return this;
        }

        public Builder withDefaultServerConfiguration(@Nullable ServerConfiguration defaultServerConfiguration) {
            this.defaultServerConfiguration = defaultServerConfiguration;
            return this;
        }

        public Builder withUnsafeCommunication(boolean unsafeCommunication) {
            this.unsafeCommunication = unsafeCommunication;
            return this;
        }

        public Builder withBinaryTransferSupport(boolean binaryTransferSupport) {
            this.binaryTransferSupport = binaryTransferSupport;
            return this;
        }

        public Builder withBinaryDatabaseInjector() {
            this.addBinaryDatabaseInjector = true;
            return this;
        }

        public Builder withBinaryTransferSupport() {
            return this.withBinaryTransferSupport(true);
        }

        public Builder withCustomBinaryDatabaseProvider(@Nullable BinaryDatabaseProvider customBinaryDatabaseProvider) {
            this.customBinaryDatabaseProvider = customBinaryDatabaseProvider;
            return this;
        }

        public Builder addFeature(@NonNull String featureDescription, @NonNull BindableService featureService) {
            if (featureDescription == null) {
                throw new NullPointerException("featureDescription is marked non-null but is null");
            }
            if (featureService == null) {
                throw new NullPointerException("featureService is marked non-null but is null");
            }
            this.featureDefinitions.put(FeatureGenerator.generateFullyQualifiedIdentifier((Feature)FeatureGenerator.generateFeature((String)featureDescription)), featureDescription);
            this.bindableServices.add(featureService);
            return this;
        }

        public Builder addFCPAffectedByMetadata(@NonNull String FCPAffectedByMetadata, String ... FQIMetadata) {
            if (FCPAffectedByMetadata == null) {
                throw new NullPointerException("FCPAffectedByMetadata is marked non-null but is null");
            }
            if (FQIMetadata == null) {
                throw new NullPointerException("FQIMetadata is marked non-null but is null");
            }
            boolean isValidFQI = Stream.of(FullyQualifiedIdentifierUtils.FullyQualifiedFeatureIdentifierPattern, FullyQualifiedIdentifierUtils.FullyQualifiedCommandIdentifierPattern, FullyQualifiedIdentifierUtils.FullyQualifiedPropertyIdentifierPattern).anyMatch(p -> p.matcher(FCPAffectedByMetadata).matches());
            if (!isValidFQI) {
                throw new RuntimeException("A valid FCP must be provided.");
            }
            for (String metadata : FQIMetadata) {
                if (!FullyQualifiedIdentifierUtils.FullyQualifiedMetadataIdentifierPattern.matcher(metadata).matches()) {
                    throw new RuntimeException("A valid fully qualified metadate identifier must be provided.");
                }
                this.affectedByMetadata.put((Object)FCPAffectedByMetadata, (Object)metadata);
            }
            return this;
        }

        public Builder removeFCPAffectedByMetadata(@NonNull String FCPAffectedByMetadata, String ... FQIMetadata) {
            if (FCPAffectedByMetadata == null) {
                throw new NullPointerException("FCPAffectedByMetadata is marked non-null but is null");
            }
            if (FQIMetadata == null) {
                throw new NullPointerException("FQIMetadata is marked non-null but is null");
            }
            for (String metadata : FQIMetadata) {
                this.affectedByMetadata.remove((Object)FCPAffectedByMetadata, (Object)metadata);
            }
            return this;
        }

        public Builder addFeature(@NonNull FeatureImplementation featureImplementation) {
            if (featureImplementation == null) {
                throw new NullPointerException("featureImplementation is marked non-null but is null");
            }
            this.featureDefinitions.put(FeatureGenerator.generateFullyQualifiedIdentifier((Feature)FeatureGenerator.generateFeature((String)featureImplementation.getFeatureDescription())), featureImplementation.getFeatureDescription());
            this.bindableServices.add(featureImplementation.getService());
            return this;
        }

        public Builder addFeature(@NonNull String featureDescription, @NonNull ServerServiceDefinition serverServiceFeatureDefinition) {
            if (featureDescription == null) {
                throw new NullPointerException("featureDescription is marked non-null but is null");
            }
            if (serverServiceFeatureDefinition == null) {
                throw new NullPointerException("serverServiceFeatureDefinition is marked non-null but is null");
            }
            this.featureDefinitions.put(FeatureGenerator.generateFullyQualifiedIdentifier((Feature)FeatureGenerator.generateFeature((String)featureDescription)), featureDescription);
            this.serverServices.add(serverServiceFeatureDefinition);
            return this;
        }

        public SiLAServer start() throws IOException {
            if (this.port == null) {
                this.port = SocketUtils.getAvailablePortInRange((int)defaultPortRange[0], (int)defaultPortRange[1]);
            }
            return new SiLAServer(this);
        }

        public Builder withTLS(@NonNull X509Certificate certChain, @NonNull PrivateKey privateKey) {
            if (certChain == null) {
                throw new NullPointerException("certChain is marked non-null but is null");
            }
            if (privateKey == null) {
                throw new NullPointerException("privateKey is marked non-null but is null");
            }
            if (this.certificateFile != null || this.privateKeyFile != null) {
                throw new RuntimeException("Cannot use Persistent TLS and Runtime TLS");
            }
            this.certificate = certChain;
            this.privateKey = privateKey;
            return this;
        }

        public Builder withPersistentTLS(@NonNull Path privateKeyFile, @NonNull Path certificateFile, @Nullable String certificatePassword) {
            if (privateKeyFile == null) {
                throw new NullPointerException("privateKeyFile is marked non-null but is null");
            }
            if (certificateFile == null) {
                throw new NullPointerException("certificateFile is marked non-null but is null");
            }
            if (this.certificate != null || this.privateKey != null) {
                throw new RuntimeException("Cannot use Runtime TLS and Persisted TLS");
            }
            this.privateKeyFile = privateKeyFile;
            this.certificateFile = certificateFile;
            this.certificatePassword = certificatePassword;
            return this;
        }

        public Builder withPersistentTLS(@NonNull Path privateKeyFile, @NonNull Path certificateFile) {
            if (privateKeyFile == null) {
                throw new NullPointerException("privateKeyFile is marked non-null but is null");
            }
            if (certificateFile == null) {
                throw new NullPointerException("certificateFile is marked non-null but is null");
            }
            return this.withPersistentTLS(privateKeyFile, certificateFile, null);
        }

        public Builder addInterceptor(@NonNull ServerInterceptor interceptor) {
            if (interceptor == null) {
                throw new NullPointerException("interceptor is marked non-null but is null");
            }
            this.interceptors.add(interceptor);
            return this;
        }

        public Builder withCustomConfigWrapperProvider(@Nullable ConfigWrapperProvider customConfigWrapperProvider) {
            this.customServerConfigProvider = customConfigWrapperProvider;
            return this;
        }

        public Builder withPersistentConfig(boolean usePersistentConfig) {
            this.persistedConfig = usePersistentConfig;
            return this;
        }

        public Builder withPersistentConfig() {
            return this.withPersistentConfig(true);
        }

        public Builder withPersistentConfigFile(Path persistentConfigFile) {
            this.persistentConfigFile = persistentConfigFile;
            this.persistedConfig = true;
            return this;
        }

        public Builder withMetadataExtractingInterceptor() {
            return this.addInterceptor(new MetadataExtractingInterceptor());
        }

        public static Builder newBuilder(@NonNull ServerInformation serverInformation) throws IOException {
            if (serverInformation == null) {
                throw new NullPointerException("serverInformation is marked non-null but is null");
            }
            return new Builder(serverInformation);
        }

        public IServerConfigWrapper getNewServerConfigurationWrapper() throws IOException {
            ServerConfiguration serverConfiguration = new ServerConfiguration(this.name != null ? this.name : this.defaultServerConfiguration.getName(), this.uuid != null ? this.uuid : this.defaultServerConfiguration.getUuid());
            IServerConfigWrapper serverConfigWrapper = this.customServerConfigProvider != null ? this.customServerConfigProvider.get(this.persistentConfigFile, serverConfiguration) : (this.persistedConfig ? new PersistentServerConfigWrapper(this.persistentConfigFile, serverConfiguration) : new NonPersistentServerConfigWrapper(serverConfiguration));
            if (this.persistedConfig && (this.uuid != null || this.name != null)) {
                serverConfigWrapper.setConfig(serverConfiguration);
            }
            return serverConfigWrapper;
        }

        private BinaryDatabase getNewBinaryDatabase(@NonNull UUID serverUUID) {
            if (serverUUID == null) {
                throw new NullPointerException("serverUUID is marked non-null but is null");
            }
            try {
                if (this.customBinaryDatabaseProvider != null) {
                    return this.customBinaryDatabaseProvider.get(serverUUID);
                }
                return new H2BinaryDatabase(serverUUID);
            }
            catch (SQLException e) {
                log.warn("Error while setting BinaryDatabase: {}", (Object)e.getMessage(), (Object)e);
                throw new RuntimeException(e);
            }
        }

        private Builder(@NonNull ServerInformation serverInformation) {
            if (serverInformation == null) {
                throw new NullPointerException("serverInformation is marked non-null but is null");
            }
            this.serverInformation = serverInformation;
            this.interceptors.add(TransmitThrowableInterceptor.instance());
            this.defaultServerConfiguration = new ServerConfiguration(serverInformation.getType());
        }

        public ConfigWrapperProvider getCustomServerConfigProvider() {
            return this.customServerConfigProvider;
        }

        public BinaryDatabaseProvider getCustomBinaryDatabaseProvider() {
            return this.customBinaryDatabaseProvider;
        }

        @FunctionalInterface
        public static interface BinaryDatabaseProvider {
            public BinaryDatabase get(@NonNull UUID var1) throws SQLException;
        }

        @FunctionalInterface
        public static interface ConfigWrapperProvider {
            public IServerConfigWrapper get(@NonNull Path var1, @NonNull ServerConfiguration var2);
        }
    }
}

