/*
 * Decompiled with CFR 0.152.
 */
package sila_java.library.server_base.binary_transfer.database.impl;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;
import java.time.Duration;
import java.time.OffsetDateTime;
import java.util.UUID;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import lombok.NonNull;
import org.h2.jdbcx.JdbcDataSource;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import sila_java.library.core.sila.binary_transfer.BinaryInfo;
import sila_java.library.server_base.binary_transfer.Binary;
import sila_java.library.server_base.binary_transfer.database.BinaryDatabase;
import sila_java.library.server_base.binary_transfer.database.BinaryDatabaseException;

public class H2BinaryDatabase
implements BinaryDatabase {
    private static final Logger log = LoggerFactory.getLogger(H2BinaryDatabase.class);
    private static final String SQL_INIT = "CREATE TABLE IF NOT EXISTS BINARIES\n(\n    ID UUID NOT NULL PRIMARY KEY,\n    PARAMETER_ID VARCHAR NOT NULL,\n    EXPIRATION TIMESTAMP WITH TIME ZONE NOT NULL ,\n    DATA BLOB NOT NULL\n);\n";
    private static final String ADD_PARAMETER_ID_IF_MISSING = "ALTER TABLE BINARIES ADD COLUMN IF NOT EXISTS PARAMETER_ID VARCHAR;\n";
    private static final String DB_USER = "sa";
    private static final String DB_PASS = "sa";
    private static final int CLEANUP_CHECK_INTERVAL_SEC = 300;
    private static final Duration BINARY_EXTEND_DURATION = Duration.ofMinutes(10L);
    private final ScheduledExecutorService cleaner = Executors.newSingleThreadScheduledExecutor();
    private final Connection connection;

    public H2BinaryDatabase(@NonNull UUID serverId) throws SQLException {
        if (serverId == null) {
            throw new NullPointerException("serverId is marked non-null but is null");
        }
        JdbcDataSource ds = new JdbcDataSource();
        String binaryDbPath = "jdbc:h2:~/.sila/binary/" + serverId + ";DB_CLOSE_ON_EXIT=FALSE";
        ds.setURL(binaryDbPath);
        log.info("binary DB persisted at: {}", (Object)binaryDbPath);
        ds.setUser("sa");
        ds.setPassword("sa");
        this.connection = ds.getConnection();
        try (Statement statement = this.connection.createStatement();){
            statement.addBatch(SQL_INIT);
            statement.addBatch(ADD_PARAMETER_ID_IF_MISSING);
            statement.executeBatch();
        }
        catch (SQLException e) {
            this.connection.close();
            throw e;
        }
        this.cleaner.scheduleAtFixedRate(() -> {
            try {
                this.purgeExpiredBinaries();
            }
            catch (BinaryDatabaseException e) {
                log.warn("Following exception occurred while purging expired binaries: {}", (Object)e.getMessage(), (Object)e);
            }
        }, 0L, 300L, TimeUnit.SECONDS);
    }

    @Override
    public void close() {
        if (!this.cleaner.isTerminated() && !this.cleaner.isShutdown()) {
            this.cleaner.shutdownNow();
        }
        try {
            if (this.connection.isClosed()) {
                log.info("Connection is already closed");
                return;
            }
        }
        catch (SQLException e) {
            log.warn("Failed to check connection state", (Throwable)e);
        }
        try {
            this.purgeExpiredBinaries();
        }
        catch (BinaryDatabaseException e) {
            log.warn("Error occurred while removing binaries from database: {}", (Object)e.getMessage(), (Object)e);
        }
        try {
            this.connection.close();
        }
        catch (SQLException e) {
            log.warn("Exception occurred while closing db connection", (Throwable)e);
        }
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    @Override
    public Binary getBinary(@NonNull UUID binaryTransferUuid) throws BinaryDatabaseException {
        if (binaryTransferUuid == null) {
            throw new NullPointerException("binaryTransferUuid is marked non-null but is null");
        }
        String query = "SELECT ID, PARAMETER_ID, EXPIRATION, DATA, octet_length(BINARIES.DATA) AS BYTE_SIZE FROM BINARIES WHERE ID = ?";
        try (PreparedStatement preparedStatement = this.connection.prepareStatement("SELECT ID, PARAMETER_ID, EXPIRATION, DATA, octet_length(BINARIES.DATA) AS BYTE_SIZE FROM BINARIES WHERE ID = ?");){
            preparedStatement.setString(1, binaryTransferUuid.toString());
            preparedStatement.execute();
            ResultSet resultSet = preparedStatement.getResultSet();
            if (!resultSet.first()) {
                throw new BinaryDatabaseException("No binary found with id " + binaryTransferUuid);
            }
            Binary binary = new Binary(resultSet.getBlob("DATA"), H2BinaryDatabase.getBinaryInfo(resultSet));
            return binary;
        }
        catch (SQLException e) {
            throw new BinaryDatabaseException(e);
        }
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    @Override
    public boolean hasBinary(UUID binaryTransferUuid) throws BinaryDatabaseException {
        String query = "SELECT ID FROM BINARIES WHERE ID = ?";
        try (PreparedStatement preparedStatement = this.connection.prepareStatement("SELECT ID FROM BINARIES WHERE ID = ?");){
            preparedStatement.setString(1, binaryTransferUuid.toString());
            preparedStatement.execute();
            boolean bl = preparedStatement.getResultSet().first();
            return bl;
        }
        catch (SQLException e) {
            throw new BinaryDatabaseException(e);
        }
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    @Override
    public BinaryInfo getBinaryInfo(@NonNull UUID binaryTransferUuid) throws BinaryDatabaseException {
        if (binaryTransferUuid == null) {
            throw new NullPointerException("binaryTransferUuid is marked non-null but is null");
        }
        String query = "SELECT ID, PARAMETER_ID, EXPIRATION, octet_length(BINARIES.DATA) AS BYTE_SIZE FROM BINARIES WHERE ID = ?";
        try (PreparedStatement preparedStatement = this.connection.prepareStatement("SELECT ID, PARAMETER_ID, EXPIRATION, octet_length(BINARIES.DATA) AS BYTE_SIZE FROM BINARIES WHERE ID = ?");){
            preparedStatement.setString(1, binaryTransferUuid.toString());
            preparedStatement.execute();
            ResultSet resultSet = preparedStatement.getResultSet();
            if (!resultSet.first()) {
                throw new BinaryDatabaseException("No blob found with id " + binaryTransferUuid);
            }
            BinaryInfo binaryInfo = H2BinaryDatabase.getBinaryInfo(resultSet);
            return binaryInfo;
        }
        catch (SQLException e) {
            throw new BinaryDatabaseException(e);
        }
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    @Override
    public Duration addBinary(@NonNull UUID binaryTransferUuid, byte[] bytes, @NonNull String parameterIdentifier) throws BinaryDatabaseException {
        if (binaryTransferUuid == null) {
            throw new NullPointerException("binaryTransferUuid is marked non-null but is null");
        }
        if (parameterIdentifier == null) {
            throw new NullPointerException("parameterIdentifier is marked non-null but is null");
        }
        try (ByteArrayInputStream byteStream = new ByteArrayInputStream(bytes);){
            Duration duration = this.addBinary(binaryTransferUuid, byteStream, parameterIdentifier);
            return duration;
        }
        catch (IOException e) {
            throw new RuntimeException(e);
        }
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    @Override
    public Duration addBinary(@NonNull UUID binaryTransferUuid, @NonNull InputStream stream, @NonNull String parameterIdentifier) throws BinaryDatabaseException {
        if (binaryTransferUuid == null) {
            throw new NullPointerException("binaryTransferUuid is marked non-null but is null");
        }
        if (stream == null) {
            throw new NullPointerException("stream is marked non-null but is null");
        }
        if (parameterIdentifier == null) {
            throw new NullPointerException("parameterIdentifier is marked non-null but is null");
        }
        String query = "INSERT INTO BINARIES (ID, PARAMETER_ID, EXPIRATION, DATA) VALUES (?, ?, ?, ?)";
        try (PreparedStatement preparedStatement = this.connection.prepareStatement("INSERT INTO BINARIES (ID, PARAMETER_ID, EXPIRATION, DATA) VALUES (?, ?, ?, ?)");){
            preparedStatement.setObject(1, binaryTransferUuid);
            OffsetDateTime expirationDate = OffsetDateTime.now().plus(BINARY_EXTEND_DURATION);
            preparedStatement.setObject(2, parameterIdentifier);
            preparedStatement.setObject(3, expirationDate);
            preparedStatement.setBinaryStream(4, stream);
            preparedStatement.execute();
            Duration duration = Duration.between(OffsetDateTime.now(), expirationDate);
            return duration;
        }
        catch (SQLException e) {
            throw new BinaryDatabaseException(e);
        }
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    @Override
    public Duration extendBinaryExpiration(@NonNull UUID binaryTransferUuid) throws BinaryDatabaseException {
        if (binaryTransferUuid == null) {
            throw new NullPointerException("binaryTransferUuid is marked non-null but is null");
        }
        String query = "UPDATE BINARIES SET EXPIRATION = ? WHERE ID = ?";
        try (PreparedStatement preparedStatement = this.connection.prepareStatement("UPDATE BINARIES SET EXPIRATION = ? WHERE ID = ?");){
            OffsetDateTime newExpiration = this.getBinaryInfo(binaryTransferUuid).getExpiration().plus(BINARY_EXTEND_DURATION);
            preparedStatement.setObject(1, newExpiration);
            preparedStatement.setObject(2, binaryTransferUuid);
            preparedStatement.execute();
            Duration duration = Duration.between(OffsetDateTime.now(), newExpiration);
            return duration;
        }
        catch (SQLException e) {
            throw new BinaryDatabaseException(e);
        }
    }

    @Override
    public void removeAllBinaries() throws BinaryDatabaseException {
        String query = "delete from BINARIES";
        try (PreparedStatement preparedStatement = this.connection.prepareStatement("delete from BINARIES");){
            preparedStatement.execute();
        }
        catch (SQLException e) {
            throw new BinaryDatabaseException(e);
        }
    }

    @Override
    public void removeBinary(@NonNull UUID binaryTransferUuid) throws BinaryDatabaseException {
        if (binaryTransferUuid == null) {
            throw new NullPointerException("binaryTransferUuid is marked non-null but is null");
        }
        String query = "delete from BINARIES where ID = ?";
        try (PreparedStatement preparedStatement = this.connection.prepareStatement("delete from BINARIES where ID = ?");){
            preparedStatement.setObject(1, binaryTransferUuid);
            preparedStatement.execute();
        }
        catch (SQLException e) {
            throw new BinaryDatabaseException(e);
        }
    }

    private void purgeExpiredBinaries() throws BinaryDatabaseException {
        String query = "delete from BINARIES where EXPIRATION < ?";
        try (PreparedStatement preparedStatement = this.connection.prepareStatement("delete from BINARIES where EXPIRATION < ?");){
            preparedStatement.setObject(1, OffsetDateTime.now());
            int nbRemoved = preparedStatement.executeUpdate();
            if (nbRemoved > 0) {
                log.info("Purged {} expired binary.", (Object)nbRemoved);
            }
        }
        catch (SQLException e) {
            throw new BinaryDatabaseException(e);
        }
    }

    private static BinaryInfo getBinaryInfo(ResultSet resultSet) throws SQLException {
        UUID id = UUID.fromString(resultSet.getString("ID"));
        OffsetDateTime expiration = resultSet.getObject("EXPIRATION", OffsetDateTime.class);
        String parameterId = resultSet.getObject("PARAMETER_ID", String.class);
        long byteSize = resultSet.getLong("BYTE_SIZE");
        return new BinaryInfo(id, expiration, parameterId, byteSize);
    }
}

