package sila_java.library.server_base;

import io.grpc.*;
import lombok.Getter;
import lombok.NonNull;
import lombok.SneakyThrows;
import lombok.extern.slf4j.Slf4j;
import sila_java.library.core.discovery.SiLAServerRegistration;
import sila_java.library.core.encryption.SelfSignedCertificate;
import sila_java.library.core.sila.mapping.feature.FeatureGenerator;
import sila_java.library.core.sila.mapping.feature.MalformedSiLAFeature;
import sila_java.library.server_base.config.IServerConfigWrapper;
import sila_java.library.server_base.config.NonPersistentServerConfigWrapper;
import sila_java.library.server_base.config.PersistentServerConfigWrapper;
import sila_java.library.server_base.identification.ServerInformation;
import sila_java.library.server_base.binary_transfer.database.BinaryDatabase;
import sila_java.library.server_base.binary_transfer.database.impl.H2BinaryDatabase;
import sila_java.library.server_base.binary_transfer.download.DownloadService;
import sila_java.library.server_base.binary_transfer.upload.UploadService;
import sila_java.library.server_base.metadata.MetadataExtractingInterceptor;
import sila_java.library.server_base.standard_features.FeatureImplementation;
import sila_java.library.server_base.standard_features.SiLAServiceServer;
import sila_java.library.server_base.utils.TransmitThrowableInterceptor;

import javax.annotation.Nullable;
import java.io.*;
import java.nio.file.Files;
import java.nio.file.Path;
import java.security.*;
import java.security.cert.*;
import java.sql.SQLException;
import java.util.*;
import java.util.concurrent.TimeUnit;

import static sila_java.library.core.encryption.EncryptionUtils.*;
import static sila_java.library.core.utils.SocketUtils.getAvailablePortInRange;

/**
 * SiLA Server
 */
@Slf4j
public class SiLAServer implements AutoCloseable {
    private final static int SHUTDOWN_TIMEOUT = 20; // [s]
    private final SiLAServerRegistration serverRegistration;
    private final Server server;
    private final BinaryDatabase binaryDatabaseImpl;
    @Getter
    private final Optional<DownloadService> downloadService;
    @Getter
    private final Optional<UploadService> uploadService;

    /**
     * Create and start a Base Server to expose the SiLA Features
     *
     * @param builder The builder
     *
     * @implNote Instance can only withoutConfig one server at a time.
     */
    private SiLAServer(@NonNull final Builder builder) throws IOException {
        final SiLAServiceServer siLAServiceServer = new SiLAServiceServer(
                builder.serverConfig,
                builder.serverInformation,
                builder.featureDefinitions
        );
        log.info("Server registered on SiLAService with features={}", builder.featureDefinitions.keySet());

        final UUID serverUuid = builder.serverConfig.getCacheConfig().getUuid();

        if (builder.privateKeyFile != null && builder.certificateFile != null) {
            final File privateKeyFile = builder.privateKeyFile.toFile();
            final File certificateFile = builder.certificateFile.toFile();
            final boolean privateKeyFileExist = Files.exists(builder.privateKeyFile);
            final boolean certificateFileExist = Files.exists(builder.certificateFile);
            if (!privateKeyFileExist && certificateFileExist) {
                throw new RuntimeException("Certificate was provided, but couldn't find Private Key");
            }
            if (!certificateFileExist && privateKeyFileExist) {
                throw new RuntimeException("Private Key was provided, but couldn't find Certificate");
            }
            if (privateKeyFileExist && certificateFileExist) {
                builder.privateKey = readPrivateKey(privateKeyFile, builder.certificatePassword);
                builder.certificate = readCertificate(certificateFile);
            } else {
                privateKeyFile.getParentFile().mkdirs();
                privateKeyFile.createNewFile();
                certificateFile.getParentFile().mkdirs();
                certificateFile.createNewFile();
            }
        }

        if (builder.unsafeCommunication) {
            log.warn("Using plain-text communication forbidden by the SiLA 2 Standard !!");
        } else if (builder.certificate == null && builder.privateKey == null) {
            log.info("No certificate provided, creating and using self signed certificate");
            final SelfSignedCertificate selfSignedCertificate;
            try {
                String serverIp = null;
                if (builder.interfaceName != null) {
                    try (final SiLAServerRegistration tmpRegsitration = new SiLAServerRegistration(
                            serverUuid,
                            builder.interfaceName,
                            builder.port
                    )) {
                        serverIp = tmpRegsitration.findInetAddress().getHostAddress();
                    } catch (Exception e) {
                        log.warn("Failed to retrieve server ip for discovery!", e);
                    }
                }
                log.warn("Creating self signed certificate for server uuid {} and ip {}", serverUuid, serverIp);
                selfSignedCertificate = SelfSignedCertificate
                        .newBuilder()
                        .withServerUUID(serverUuid)
                        .withServerIP(serverIp)
                        .build();
            } catch (SelfSignedCertificate.CertificateGenerationException e) {
                throw new RuntimeException(e);
            }
            builder.certificate = selfSignedCertificate.getCertificate();
            builder.privateKey = selfSignedCertificate.getPrivateKey();
            if (builder.certificateFile != null) {
                writeCertificateToFile(builder.certificateFile.toFile(), builder.certificate);
                log.info("Wrote certificate to {}", builder.certificateFile);
            }
            if (builder.privateKeyFile != null) {
                log.info("Wrote private key to {}", builder.privateKeyFile);
                writePrivateKeyToFile(builder.privateKeyFile.toFile(), builder.privateKey);
            }
        }

        final ServerCredentials serverCredentials;
        // If cert and key we use transport security
        if (builder.certificate != null && builder.privateKey != null && !builder.unsafeCommunication) {
            try (InputStream certChainStream = certificateToStream(builder.certificate);
                 InputStream keyStream = keyToStream(builder.privateKey)) {
                serverCredentials = TlsServerCredentials.create(certChainStream, keyStream);
            } catch (final CertificateException e) {
                throw new IOException(e);
            }
            log.info("Server will use safe encrypted communication.");
        } else {
            serverCredentials = InsecureServerCredentials.create();
            log.warn("Server will use deprecated unsafe plain-text communication.");
        }

        final ServerBuilder<?> serverBuilder = Grpc.newServerBuilderForPort(builder.port, serverCredentials)
                .addService(siLAServiceServer.getService());

        this.binaryDatabaseImpl = builder.binaryDatabaseImpl;

        if (this.binaryDatabaseImpl == null) {
            log.warn("Server will not support binary transfer");
            this.downloadService = Optional.empty();
            this.uploadService = Optional.empty();
        } else {
            log.info("Server will support binary transfer");
            this.downloadService = Optional.of(new DownloadService(this.binaryDatabaseImpl));
            this.uploadService = Optional.of(new UploadService(this.binaryDatabaseImpl));
            serverBuilder.addService(this.downloadService.get());
            serverBuilder.addService(this.uploadService.get());
        }

        builder.bindableServices.forEach(serverBuilder::addService);
        builder.serverServices.forEach(serverBuilder::addService);

        if (builder.interfaceName != null) {
            serverRegistration = new SiLAServerRegistration(
                    serverUuid,
                    builder.interfaceName,
                    builder.port,
                    (builder.certificate != null && !builder.unsafeCommunication) ? writeCertificateToString(builder.certificate) : null
            );
            log.info("Server registering with discovery.");
        } else {
            serverRegistration = null;
            log.warn("Server starting without specifying a network interface, discovery is not enabled.");
        }

        builder.interceptors.forEach(serverInterceptor -> {
            serverBuilder.intercept(serverInterceptor);
            log.info("Added interceptor of type " + serverInterceptor.getClass().getTypeName());
        });

        this.server = serverBuilder.build().start();

        log.info("Server started on port={}", builder.port);
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void close() {
        log.info("[stop] stopping server...");
        if (this.serverRegistration != null) {
            this.serverRegistration.close();
        }

        // Stop server
        if (this.server != null && !this.server.isTerminated() && !this.server.isShutdown()) {
            try {
                log.info("[stop] stopping the server ...");
                this.server.shutdownNow().awaitTermination(SHUTDOWN_TIMEOUT, TimeUnit.SECONDS);
                log.info("[stop] the server was stopped");
            } catch (InterruptedException e) {
                log.warn("[stop] could not shutdown the server within {} seconds", SHUTDOWN_TIMEOUT);
            }
            log.info("[stop] stopped");
        } else {
            log.info("[stop] server already stopped");
        }
        if (this.binaryDatabaseImpl != null) {
            try {
                this.binaryDatabaseImpl.close();
            } catch (Exception e) {
                log.warn("Error occurred while closing binary database {}", e.getMessage(), e);
            }
        }
    }

    /**
     * Server Builder
     */
    public static class Builder {
        private static int[] defaultPortRange = {50052, 50052+256};

        // Mandatory Arguments
        private final ServerInformation serverInformation;
        @Getter
        private final IServerConfigWrapper serverConfig;
        private final Map<String, String> featureDefinitions = new HashMap<>();
        private final List<BindableService> bindableServices = new ArrayList<>();
        private final List<ServerServiceDefinition> serverServices = new ArrayList<>();

        // Optional Arguments
        private String interfaceName = null;
        private Integer port = null;
        private BinaryDatabase binaryDatabaseImpl = null;

        // TLS
        private Path certificateFile = null;
        private String certificatePassword;

        private Path privateKeyFile = null;
        private X509Certificate certificate = null;
        private PrivateKey privateKey = null;

        private boolean unsafeCommunication = false;
        private final List<ServerInterceptor> interceptors = new ArrayList<>();

        public Map<String, String> getFeatureDefinitions() {
            return Collections.unmodifiableMap(this.featureDefinitions);
        }

        /**
         * Enables Discovery on a certain interface
         * @param interfaceName  Name of network interface to use discovery
         */
        public Builder withDiscovery(@NonNull final String interfaceName) {
            this.interfaceName = interfaceName;
            return this;
        }

        /**
         * Define Specific Port for the services (otherwise a default range will be chosen)
         * @param port Port on which the server runs.
         */
        public Builder withPort(int port) {
            this.port = port;
            return this;
        }

        /**
         * Enable support for binary transfer
         */
        public Builder withBinaryTransferSupport(@NonNull final BinaryDatabase binarySupportImpl) {
            this.binaryDatabaseImpl = binarySupportImpl;
            return this;
        }

        /**
         * @deprecated Unsafe plain-text communication is forbidden by the SiLA 2 Standard
         */
        public Builder withUnsafeCommunication(final boolean unsafeCommunication) {
            this.unsafeCommunication = unsafeCommunication;
            return this;
        }

        /**
         * Enable support for binary transfer by creating and using the default H2 binary transfer database implementation
         * @see H2BinaryDatabase
         * @return the created binary database
         */
        @SneakyThrows
        public BinaryDatabase withBinaryTransferSupport() {
            try {
                if (binaryDatabaseImpl != null) {
                    binaryDatabaseImpl.close();
                    binaryDatabaseImpl = null;
                    log.warn("Duplicated call to withBinaryTransferSupport, closing previous database.");
                }
                binaryDatabaseImpl = new H2BinaryDatabase(this.serverConfig.getCacheConfig().getUuid());
            } catch (SQLException | IOException  e) {
                log.warn("Error while setting BinaryDatabase: {}", e.getMessage(), e);
                throw new RuntimeException(e);
            }
            return binaryDatabaseImpl;
        }

        /**
         * Add a Feature to be exposed
         * @param featureDescription Feature Description as String Content in XML
         * @param featureService Feature Service implemented with gRPC
         *
         * @implNote Unfortunately there is no straightforward way to validate the pure protobuf
         * descriptions and the generated gRPC descriptions.
         */
        @SneakyThrows({IOException.class, MalformedSiLAFeature.class})
        public Builder addFeature(
                @NonNull final String featureDescription,
                @NonNull final BindableService featureService
        ) {
            featureDefinitions.put(
                    FeatureGenerator.generateFullyQualifiedIdentifier(
                            FeatureGenerator.generateFeature(featureDescription)
                    ),
                    featureDescription
            );
            bindableServices.add(featureService);
            return this;
        }

        /**
         * Add a Feature to be exposed
         * @param featureImplementation Exposing both the description and implementation
         */
        @SneakyThrows({IOException.class, MalformedSiLAFeature.class})
        public Builder addFeature(@NonNull final FeatureImplementation featureImplementation) {
            featureDefinitions.put(
                    FeatureGenerator.generateFullyQualifiedIdentifier(
                            FeatureGenerator.generateFeature(featureImplementation.getFeatureDescription())
                    ),
                    featureImplementation.getFeatureDescription()
            );
            bindableServices.add(featureImplementation.getService());
            return this;
        }

        /**
         * Add a Feature to be exposed
         * @param featureDescription Feature Description as String Content in XML
         * @param serverServiceFeatureDefinition Server service definition feature implemented with gRPC
         *
         */
        @SneakyThrows({IOException.class, MalformedSiLAFeature.class})
        public Builder addFeature(
                @NonNull final String featureDescription,
                @NonNull final ServerServiceDefinition serverServiceFeatureDefinition
        ) {
            featureDefinitions.put(
                    FeatureGenerator.generateFullyQualifiedIdentifier(
                            FeatureGenerator.generateFeature(featureDescription)
                    ),
                    featureDescription
            );
            serverServices.add(serverServiceFeatureDefinition);
            return this;
        }

        /**
         * Starts and Creates the SiLA Server
         * @return SiLA Server
         */
        public SiLAServer start() throws IOException {
            if (this.port == null) {
                this.port = getAvailablePortInRange(defaultPortRange[0], defaultPortRange[1]);
            }

            return new SiLAServer(this);
        }

        /**
         * Use TLS certification
         * @param certChain InputStream certification
         * @param privateKey InputStream private key
         */
        public Builder withTLS(
                @NonNull final X509Certificate certChain,
                @NonNull final PrivateKey privateKey
        ) {
            if (this.certificateFile != null || this.privateKeyFile != null) {
                throw new RuntimeException("Cannot use Persisted TLS and Runtime TLS");
            }
            this.certificate = certChain;
            this.privateKey = privateKey;
            return this;
        }

        /**
         * Use Persisted TLS certification. Will create a self-signed certificate if both file are missing
         * @param certificateFile Path to the private key PEM file
         * @param privateKeyFile Path to the certificate PEM file
         * @param certificatePassword Password of the certificate if any
         */
        public Builder withPersistedTLS(
                @NonNull final Path privateKeyFile,
                @NonNull final Path certificateFile,
                @Nullable final String certificatePassword
        ) {
            if (this.certificate != null || this.privateKey != null) {
                throw new RuntimeException("Cannot use Runtime TLS and Persisted TLS");
            }
            this.privateKeyFile = privateKeyFile;
            this.certificateFile = certificateFile;
            this.certificatePassword = certificatePassword;
            return this;
        }

        /**
         * Add a gRPC {@link ServerInterceptor} to the server
         *
         * @param interceptor The {@link ServerInterceptor} to add
         * @return this builder
         */
        public Builder addInterceptor(@NonNull final ServerInterceptor interceptor) {
            this.interceptors.add(interceptor);
            return this;
        }

        /**
         * Add a {@link MetadataExtractingInterceptor} to the server
         *
         * @return this builder
         *
         * @implNote The order of added interceptors is preserved. Thus, interceptors added with
         * {@link SiLAServer.Builder#addInterceptor(ServerInterceptor)} before this method precede the
         * {@link MetadataExtractingInterceptor} in the call handling chain.
         */
        public Builder withMetadataExtractingInterceptor() {
            return addInterceptor(new MetadataExtractingInterceptor());
        }

        /**
         * Create builder for a Server with a non-persistent configuration
         *
         * @param serverInformation     Meta server information defined by the server implementer
         */
        public static Builder withoutConfig(@NonNull final ServerInformation serverInformation) {
            log.debug("Server config is non-persistent");
            return new Builder(serverInformation, new NonPersistentServerConfigWrapper(serverInformation.getType()));
        }

        /**
         * Create builder for a Server with a persistent configuration file
         *
         * @param configurationFile     The file persisting the server name and UUID data for that server instance
         * @param serverInformation     Meta server information defined by the server implementer
         */
        public static Builder withConfig(
                @NonNull final Path configurationFile,
                @NonNull final ServerInformation serverInformation
        ) throws IOException {
            log.debug("Server config is persistent");
            return new Builder(
                    serverInformation,
                    new PersistentServerConfigWrapper(configurationFile, serverInformation.getType())
            );
        }

        @SneakyThrows
        private Builder(
                @NonNull final ServerInformation serverInformation,
                @NonNull final IServerConfigWrapper serverConfig
        ) {
            this.serverInformation = serverInformation;
            this.serverConfig = serverConfig;
            this.interceptors.add(TransmitThrowableInterceptor.instance());
        }
    }
}
