/*
 * Decompiled with CFR 0.152.
 */
package sila_java.library.server_base;

import io.grpc.BindableService;
import io.grpc.Grpc;
import io.grpc.InsecureServerCredentials;
import io.grpc.Server;
import io.grpc.ServerBuilder;
import io.grpc.ServerCredentials;
import io.grpc.ServerInterceptor;
import io.grpc.ServerServiceDefinition;
import io.grpc.TlsServerCredentials;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.security.PrivateKey;
import java.security.cert.CertificateException;
import java.security.cert.X509Certificate;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.UUID;
import java.util.concurrent.TimeUnit;
import javax.annotation.Nullable;
import lombok.NonNull;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import sila_java.library.core.discovery.SiLAServerRegistration;
import sila_java.library.core.encryption.EncryptionUtils;
import sila_java.library.core.encryption.SelfSignedCertificate;
import sila_java.library.core.models.Feature;
import sila_java.library.core.sila.mapping.feature.FeatureGenerator;
import sila_java.library.core.utils.SocketUtils;
import sila_java.library.server_base.binary_transfer.database.BinaryDatabase;
import sila_java.library.server_base.binary_transfer.database.impl.H2BinaryDatabase;
import sila_java.library.server_base.binary_transfer.download.DownloadService;
import sila_java.library.server_base.binary_transfer.upload.UploadService;
import sila_java.library.server_base.config.IServerConfigWrapper;
import sila_java.library.server_base.config.NonPersistentServerConfigWrapper;
import sila_java.library.server_base.config.PersistentServerConfigWrapper;
import sila_java.library.server_base.identification.ServerInformation;
import sila_java.library.server_base.metadata.MetadataExtractingInterceptor;
import sila_java.library.server_base.standard_features.FeatureImplementation;
import sila_java.library.server_base.standard_features.SiLAServiceServer;
import sila_java.library.server_base.utils.TransmitThrowableInterceptor;

public class SiLAServer
implements AutoCloseable {
    private static final Logger log = LoggerFactory.getLogger(SiLAServer.class);
    private static final int SHUTDOWN_TIMEOUT = 20;
    private final SiLAServerRegistration serverRegistration;
    private final Server server;
    private final BinaryDatabase binaryDatabaseImpl;
    private final Optional<DownloadService> downloadService;
    private final Optional<UploadService> uploadService;

    private SiLAServer(@NonNull Builder builder) throws IOException {
        ServerCredentials serverCredentials;
        if (builder == null) {
            throw new NullPointerException("builder is marked non-null but is null");
        }
        SiLAServiceServer siLAServiceServer = new SiLAServiceServer(builder.serverConfig, builder.serverInformation, builder.featureDefinitions);
        log.info("Server registered on SiLAService with features={}", builder.featureDefinitions.keySet());
        UUID serverUuid = builder.serverConfig.getCacheConfig().getUuid();
        if (builder.privateKeyFile != null && builder.certificateFile != null) {
            File privateKeyFile = builder.privateKeyFile.toFile();
            File certificateFile = builder.certificateFile.toFile();
            boolean privateKeyFileExist = Files.exists(builder.privateKeyFile, new LinkOption[0]);
            boolean certificateFileExist = Files.exists(builder.certificateFile, new LinkOption[0]);
            if (!privateKeyFileExist && certificateFileExist) {
                throw new RuntimeException("Certificate was provided, but couldn't find Private Key");
            }
            if (!certificateFileExist && privateKeyFileExist) {
                throw new RuntimeException("Private Key was provided, but couldn't find Certificate");
            }
            if (privateKeyFileExist && certificateFileExist) {
                builder.privateKey = EncryptionUtils.readPrivateKey((File)privateKeyFile, (String)builder.certificatePassword);
                builder.certificate = EncryptionUtils.readCertificate((File)certificateFile);
            } else {
                privateKeyFile.getParentFile().mkdirs();
                privateKeyFile.createNewFile();
                certificateFile.getParentFile().mkdirs();
                certificateFile.createNewFile();
            }
        }
        if (builder.unsafeCommunication) {
            log.warn("Using plain-text communication forbidden by the SiLA 2 Standard !!");
        } else if (builder.certificate == null && builder.privateKey == null) {
            SelfSignedCertificate selfSignedCertificate;
            log.info("No certificate provided, creating and using self signed certificate");
            try {
                String serverIp = null;
                if (builder.interfaceName != null) {
                    try (SiLAServerRegistration tmpRegsitration = new SiLAServerRegistration(serverUuid, builder.interfaceName, builder.port.intValue());){
                        serverIp = tmpRegsitration.findInetAddress().getHostAddress();
                    }
                    catch (Exception e) {
                        log.warn("Failed to retrieve server ip for discovery!", (Throwable)e);
                    }
                }
                log.warn("Creating self signed certificate for server uuid {} and ip {}", (Object)serverUuid, serverIp);
                selfSignedCertificate = SelfSignedCertificate.newBuilder().withServerUUID(serverUuid).withServerIP(serverIp).build();
            }
            catch (SelfSignedCertificate.CertificateGenerationException e) {
                throw new RuntimeException(e);
            }
            builder.certificate = selfSignedCertificate.getCertificate();
            builder.privateKey = selfSignedCertificate.getPrivateKey();
            if (builder.certificateFile != null) {
                EncryptionUtils.writeCertificateToFile((File)builder.certificateFile.toFile(), (X509Certificate)builder.certificate);
                log.info("Wrote certificate to {}", (Object)builder.certificateFile);
            }
            if (builder.privateKeyFile != null) {
                log.info("Wrote private key to {}", (Object)builder.privateKeyFile);
                EncryptionUtils.writePrivateKeyToFile((File)builder.privateKeyFile.toFile(), (PrivateKey)builder.privateKey);
            }
        }
        if (builder.certificate != null && builder.privateKey != null && !builder.unsafeCommunication) {
            try (InputStream certChainStream = EncryptionUtils.certificateToStream((X509Certificate)builder.certificate);
                 InputStream keyStream = EncryptionUtils.keyToStream((PrivateKey)builder.privateKey);){
                serverCredentials = TlsServerCredentials.create((InputStream)certChainStream, (InputStream)keyStream);
            }
            catch (CertificateException e) {
                throw new IOException(e);
            }
            log.info("Server will use safe encrypted communication.");
        } else {
            serverCredentials = InsecureServerCredentials.create();
            log.warn("Server will use deprecated unsafe plain-text communication.");
        }
        ServerBuilder serverBuilder = Grpc.newServerBuilderForPort((int)builder.port, (ServerCredentials)serverCredentials).addService(siLAServiceServer.getService());
        this.binaryDatabaseImpl = builder.binaryDatabaseImpl;
        if (this.binaryDatabaseImpl == null) {
            log.warn("Server will not support binary transfer");
            this.downloadService = Optional.empty();
            this.uploadService = Optional.empty();
        } else {
            log.info("Server will support binary transfer");
            this.downloadService = Optional.of(new DownloadService(this.binaryDatabaseImpl));
            this.uploadService = Optional.of(new UploadService(this.binaryDatabaseImpl));
            serverBuilder.addService((BindableService)this.downloadService.get());
            serverBuilder.addService((BindableService)this.uploadService.get());
        }
        builder.bindableServices.forEach(arg_0 -> ((ServerBuilder)serverBuilder).addService(arg_0));
        builder.serverServices.forEach(arg_0 -> ((ServerBuilder)serverBuilder).addService(arg_0));
        if (builder.interfaceName != null) {
            this.serverRegistration = new SiLAServerRegistration(serverUuid, builder.interfaceName, builder.port.intValue(), builder.certificate != null && !builder.unsafeCommunication ? EncryptionUtils.writeCertificateToString((X509Certificate)builder.certificate) : null);
            log.info("Server registering with discovery.");
        } else {
            this.serverRegistration = null;
            log.warn("Server starting without specifying a network interface, discovery is not enabled.");
        }
        builder.interceptors.forEach(serverInterceptor -> {
            serverBuilder.intercept(serverInterceptor);
            log.info("Added interceptor of type " + serverInterceptor.getClass().getTypeName());
        });
        this.server = serverBuilder.build().start();
        log.info("Server started on port={}", (Object)builder.port);
    }

    @Override
    public void close() {
        log.info("[stop] stopping server...");
        if (this.serverRegistration != null) {
            this.serverRegistration.close();
        }
        if (this.server != null && !this.server.isTerminated() && !this.server.isShutdown()) {
            try {
                log.info("[stop] stopping the server ...");
                this.server.shutdownNow().awaitTermination(20L, TimeUnit.SECONDS);
                log.info("[stop] the server was stopped");
            }
            catch (InterruptedException e) {
                log.warn("[stop] could not shutdown the server within {} seconds", (Object)20);
            }
            log.info("[stop] stopped");
        } else {
            log.info("[stop] server already stopped");
        }
        if (this.binaryDatabaseImpl != null) {
            try {
                this.binaryDatabaseImpl.close();
            }
            catch (Exception e) {
                log.warn("Error occurred while closing binary database {}", (Object)e.getMessage(), (Object)e);
            }
        }
    }

    public Optional<DownloadService> getDownloadService() {
        return this.downloadService;
    }

    public Optional<UploadService> getUploadService() {
        return this.uploadService;
    }

    public static class Builder {
        private static int[] defaultPortRange = new int[]{50052, 50308};
        private final ServerInformation serverInformation;
        private final IServerConfigWrapper serverConfig;
        private final Map<String, String> featureDefinitions = new HashMap<String, String>();
        private final List<BindableService> bindableServices = new ArrayList<BindableService>();
        private final List<ServerServiceDefinition> serverServices = new ArrayList<ServerServiceDefinition>();
        private String interfaceName = null;
        private Integer port = null;
        private BinaryDatabase binaryDatabaseImpl = null;
        private Path certificateFile = null;
        private String certificatePassword;
        private Path privateKeyFile = null;
        private X509Certificate certificate = null;
        private PrivateKey privateKey = null;
        private boolean unsafeCommunication = false;
        private final List<ServerInterceptor> interceptors = new ArrayList<ServerInterceptor>();

        public Map<String, String> getFeatureDefinitions() {
            return Collections.unmodifiableMap(this.featureDefinitions);
        }

        public Builder withDiscovery(@NonNull String interfaceName) {
            if (interfaceName == null) {
                throw new NullPointerException("interfaceName is marked non-null but is null");
            }
            this.interfaceName = interfaceName;
            return this;
        }

        public Builder withPort(int port) {
            this.port = port;
            return this;
        }

        public Builder withBinaryTransferSupport(@NonNull BinaryDatabase binarySupportImpl) {
            if (binarySupportImpl == null) {
                throw new NullPointerException("binarySupportImpl is marked non-null but is null");
            }
            this.binaryDatabaseImpl = binarySupportImpl;
            return this;
        }

        public Builder withUnsafeCommunication(boolean unsafeCommunication) {
            this.unsafeCommunication = unsafeCommunication;
            return this;
        }

        public BinaryDatabase withBinaryTransferSupport() {
            try {
                if (this.binaryDatabaseImpl != null) {
                    this.binaryDatabaseImpl.close();
                    this.binaryDatabaseImpl = null;
                    log.warn("Duplicated call to withBinaryTransferSupport, closing previous database.");
                }
                this.binaryDatabaseImpl = new H2BinaryDatabase(this.serverConfig.getCacheConfig().getUuid());
            }
            catch (IOException | SQLException e) {
                log.warn("Error while setting BinaryDatabase: {}", (Object)e.getMessage(), (Object)e);
                throw new RuntimeException(e);
            }
            return this.binaryDatabaseImpl;
        }

        public Builder addFeature(@NonNull String featureDescription, @NonNull BindableService featureService) {
            if (featureDescription == null) {
                throw new NullPointerException("featureDescription is marked non-null but is null");
            }
            if (featureService == null) {
                throw new NullPointerException("featureService is marked non-null but is null");
            }
            this.featureDefinitions.put(FeatureGenerator.generateFullyQualifiedIdentifier((Feature)FeatureGenerator.generateFeature((String)featureDescription)), featureDescription);
            this.bindableServices.add(featureService);
            return this;
        }

        public Builder addFeature(@NonNull FeatureImplementation featureImplementation) {
            if (featureImplementation == null) {
                throw new NullPointerException("featureImplementation is marked non-null but is null");
            }
            this.featureDefinitions.put(FeatureGenerator.generateFullyQualifiedIdentifier((Feature)FeatureGenerator.generateFeature((String)featureImplementation.getFeatureDescription())), featureImplementation.getFeatureDescription());
            this.bindableServices.add(featureImplementation.getService());
            return this;
        }

        public Builder addFeature(@NonNull String featureDescription, @NonNull ServerServiceDefinition serverServiceFeatureDefinition) {
            if (featureDescription == null) {
                throw new NullPointerException("featureDescription is marked non-null but is null");
            }
            if (serverServiceFeatureDefinition == null) {
                throw new NullPointerException("serverServiceFeatureDefinition is marked non-null but is null");
            }
            this.featureDefinitions.put(FeatureGenerator.generateFullyQualifiedIdentifier((Feature)FeatureGenerator.generateFeature((String)featureDescription)), featureDescription);
            this.serverServices.add(serverServiceFeatureDefinition);
            return this;
        }

        public SiLAServer start() throws IOException {
            if (this.port == null) {
                this.port = SocketUtils.getAvailablePortInRange((int)defaultPortRange[0], (int)defaultPortRange[1]);
            }
            return new SiLAServer(this);
        }

        public Builder withTLS(@NonNull X509Certificate certChain, @NonNull PrivateKey privateKey) {
            if (certChain == null) {
                throw new NullPointerException("certChain is marked non-null but is null");
            }
            if (privateKey == null) {
                throw new NullPointerException("privateKey is marked non-null but is null");
            }
            if (this.certificateFile != null || this.privateKeyFile != null) {
                throw new RuntimeException("Cannot use Persisted TLS and Runtime TLS");
            }
            this.certificate = certChain;
            this.privateKey = privateKey;
            return this;
        }

        public Builder withPersistedTLS(@NonNull Path privateKeyFile, @NonNull Path certificateFile, @Nullable String certificatePassword) {
            if (privateKeyFile == null) {
                throw new NullPointerException("privateKeyFile is marked non-null but is null");
            }
            if (certificateFile == null) {
                throw new NullPointerException("certificateFile is marked non-null but is null");
            }
            if (this.certificate != null || this.privateKey != null) {
                throw new RuntimeException("Cannot use Runtime TLS and Persisted TLS");
            }
            this.privateKeyFile = privateKeyFile;
            this.certificateFile = certificateFile;
            this.certificatePassword = certificatePassword;
            return this;
        }

        public Builder addInterceptor(@NonNull ServerInterceptor interceptor) {
            if (interceptor == null) {
                throw new NullPointerException("interceptor is marked non-null but is null");
            }
            this.interceptors.add(interceptor);
            return this;
        }

        public Builder withMetadataExtractingInterceptor() {
            return this.addInterceptor(new MetadataExtractingInterceptor());
        }

        public static Builder withoutConfig(@NonNull ServerInformation serverInformation) {
            if (serverInformation == null) {
                throw new NullPointerException("serverInformation is marked non-null but is null");
            }
            log.debug("Server config is non-persistent");
            return new Builder(serverInformation, new NonPersistentServerConfigWrapper(serverInformation.getType()));
        }

        public static Builder withConfig(@NonNull Path configurationFile, @NonNull ServerInformation serverInformation) throws IOException {
            if (configurationFile == null) {
                throw new NullPointerException("configurationFile is marked non-null but is null");
            }
            if (serverInformation == null) {
                throw new NullPointerException("serverInformation is marked non-null but is null");
            }
            log.debug("Server config is persistent");
            return new Builder(serverInformation, new PersistentServerConfigWrapper(configurationFile, serverInformation.getType()));
        }

        private Builder(@NonNull ServerInformation serverInformation, @NonNull IServerConfigWrapper serverConfig) {
            if (serverInformation == null) {
                throw new NullPointerException("serverInformation is marked non-null but is null");
            }
            if (serverConfig == null) {
                throw new NullPointerException("serverConfig is marked non-null but is null");
            }
            this.serverInformation = serverInformation;
            this.serverConfig = serverConfig;
            this.interceptors.add(TransmitThrowableInterceptor.instance());
        }

        public IServerConfigWrapper getServerConfig() {
            return this.serverConfig;
        }
    }
}

