package sila_java.servers.file_server;

import io.grpc.stub.StreamObserver;
import lombok.NonNull;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.io.FileUtils;
import org.apache.commons.io.FilenameUtils;
import sila2.ch.unitelabs.filecreationcontroller.v1.FileCreationControllerGrpc;
import sila2.ch.unitelabs.filecreationcontroller.v1.FileCreationControllerOuterClass;
import sila_java.library.core.utils.SiLAErrors;
import sila_java.library.server_base.SiLAServerBase;
import sila_java.library.server_base.identification.ServerInformation;
import sila_java.library.server_base.utils.ArgumentHelper;

import javax.annotation.Nullable;
import java.io.*;
import java.nio.file.Path;
import java.util.HashMap;
import java.util.Map;

import static sila_java.library.core.utils.FileUtils.getResourceContent;
import static sila_java.library.core.utils.SocketUtils.getAvailablePortInRange;
import static sila_java.library.core.utils.Utils.*;

@Slf4j
public class FileServer implements Closeable {
    static final String SERVER_TYPE = "FileServer";
    static final int SERVER_PORT = 50052; // Default
    private static final int SERVER_PORT_RANGE = 256;

    private final SiLAServerBase siLAServerBase;

    public FileServer(
            @NonNull final String interfaceName,
            final int serverPort,
            final @Nullable Path config) {
        try {
            final Map<String, String> fdl = new HashMap<String, String>() {
                {
                    put(
                            "FileCreationController",
                            getResourceContent("FileCreationController.xml")
                    );
                }
            };

            final ServerInformation serverInfo = new ServerInformation(
                    SERVER_TYPE,
                    "File Server",
                    "www.unitelabs.ch",
                    "v0.0"
            );

            if (config == null) {
                this.siLAServerBase = SiLAServerBase.withoutConfig(
                        serverInfo,
                        fdl, serverPort, interfaceName,
                        new FileCreatorImpl()
                );
            } else {
                this.siLAServerBase = SiLAServerBase.withConfig(
                        config,
                        serverInfo,
                        fdl, serverPort, interfaceName,
                        new FileCreatorImpl()
                );
            }

            log.info("File Server successfully created");
        } catch (IOException e) {
            throw new RuntimeException(e.getMessage());
        }
    }

    /**
     * Simple main function that starts the server and keeps it alive
     */
    public static void main(final String[] args) {
        final ArgumentHelper argumentHelper = new ArgumentHelper(args, SERVER_TYPE);
        final int serverPort = argumentHelper.getPort() != null ?
                argumentHelper.getPort() :
                getAvailablePortInRange(SERVER_PORT, SERVER_PORT + SERVER_PORT_RANGE);

        // Start Server
        try (final FileServer server = new FileServer(
                argumentHelper.getInterface(),
                serverPort,
                argumentHelper.getConfigFile().orElse(null))
        ) {
            blockUntilStop();
        }
        System.out.println("termination complete.");
    }

    @Override
    public void close() {
        this.siLAServerBase.close();
    }

    static class FileCreatorImpl extends FileCreationControllerGrpc.FileCreationControllerImplBase {

        @Override
        public void createFile(FileCreationControllerOuterClass.CreateFile_Parameters request,
                StreamObserver<FileCreationControllerOuterClass.CreateFile_Responses> responseObserver) {
            if (!request.hasPath()) {
                responseObserver.onError(
                        SiLAErrors.generateValidationError(
                                "Path",
                                "Path parameter not set",
                                "Specify the path under which to create the file, relative to the home folder."
                        )
                );
                return;
            }

            final String path = FilenameUtils.separatorsToSystem(request.getPath().getValue());
            final String absolutePath;

            log.info("Create File called with path " + path);

            if (FilenameUtils.getPrefix(path).equals(""))
                absolutePath = FilenameUtils.concat(System.getProperty("user.home"), path);
            else {
                responseObserver.onError(
                        SiLAErrors.generateValidationError(
                                "Path",
                                "Path must be relative.",
                                "Specify a relative path (e.g. 'path/to/file.txt')"
                        )
                );
                return;
            }

            final File file = new File(absolutePath);
            try {
                FileUtils.forceMkdirParent(file);
            } catch (IOException e) {
                responseObserver.onError(
                        SiLAErrors.generateExecutionError(
                                "FileCreationFailed",
                                "Could not create parent directory",
                                "Specify another path"
                        )
                );
                return;
            }

            final String content = request.hasContent() ? request.getContent().getValue() : "";

            System.out.println(content);

            log.info("Content received: " + content);

            try {
                writeToFile(content, file);
            } catch (IOException e) {
                responseObserver.onError(
                        SiLAErrors.generateExecutionError(
                                "FileWriteFailed",
                                "Could not create or write to file",
                                "Make sure you have the correct access rights or specify another path"
                        )
                );
                return;
            }

            log.info("File created at: " + absolutePath);

            FileCreationControllerOuterClass.CreateFile_Responses result =
                    FileCreationControllerOuterClass.CreateFile_Responses.newBuilder()
                    .build();
            responseObserver.onNext(result);
            responseObserver.onCompleted();
        }

        /**
         * Private utility to preserve line breaks in file writing correctly
         */
        private static void writeToFile(String string, File file) throws IOException {
            try (
                    BufferedReader reader = new BufferedReader(new StringReader(string));
                    PrintWriter writer = new PrintWriter(new FileWriter(file))
            ) {
                reader.lines().forEach(writer::println);
            }
        }
    }

}
