package sila_java.servers.hello_sila;

import io.grpc.stub.StreamObserver;
import lombok.NonNull;
import sila2.org.silastandard.SiLAFramework;
import sila2.org.silastandard.examples.greetingprovider.v1.GreetingProviderGrpc;
import sila2.org.silastandard.examples.greetingprovider.v1.GreetingProviderOuterClass;
import sila_java.library.core.utils.SiLAErrors;
import sila_java.library.server_base.SiLAServerBase;
import sila_java.library.server_base.identification.ServerInformation;
import sila_java.library.server_base.utils.ArgumentHelper;

import javax.annotation.Nullable;
import java.io.Closeable;
import java.io.IOException;
import java.nio.file.Path;
import java.time.Year;
import java.util.HashMap;
import java.util.Map;

import static sila_java.library.core.utils.Utils.blockUntilStop;
import static sila_java.library.core.utils.FileUtils.getResourceContent;
import static sila_java.library.core.utils.SocketUtils.getAvailablePortInRange;

/**
 * Example minimal SiLA Server
 */
public class HelloSiLAServer implements Closeable {
    // Every SiLA Server needs to define a name and a port
    static final String SERVER_TYPE = "HelloSiLAServer";
    static final int SERVER_PORT = 50052; // Default
    private static final int SERVER_PORT_RANGE = 256;

    /*
     The only member needed is the SiLAServerBase for registering to discovery
     and exposing the services correctly.
      */
    private final SiLAServerBase siLAServerBase;

    /**
     * Create and start the server
     * @param interfaceName Network Interface Name to expose Discovery on
     * @param serverPort Port to serve the SiLA Services
     * @param config (optional) Configuration File Path that is used to persist UUID and Server Name
     */
    public HelloSiLAServer(
            @NonNull final String interfaceName,
            final int serverPort,
            final @Nullable Path config) {
        try {
            // Create a Map with the Feature Identifiers and Content
            final Map<String, String> fdl = new HashMap<String, String>() {
                {
                    put(
                            "GreetingProvider",
                            getResourceContent("GreetingProvider.xml")
                    );
                }
            };

            /*
             Start the minimum SiLA Server with the Meta Information
             and the gRPC Server Implementations (found below)
              */
            final ServerInformation serverInfo = new ServerInformation(
                    SERVER_TYPE,
                    "Simple Example of a SiLA Server",
                    "www.sila-standard.org",
                    "v0.0"
            );

            if (config == null) {
                this.siLAServerBase = SiLAServerBase.withoutConfig(
                        serverInfo,
                        fdl, serverPort, interfaceName,
                        new GreeterImpl()
                );
            } else {
                this.siLAServerBase = SiLAServerBase.withConfig(
                        config,
                        serverInfo,
                        fdl, serverPort, interfaceName,
                        new GreeterImpl()
                );
            }
        } catch (IOException e) {
            throw new RuntimeException(e.getMessage());
        }
    }

    @Override
    public void close() {
        this.siLAServerBase.close();
    }

    /**
     * Simple main function that starts the server and keeps it alive
     */
    public static void main(final String[] args) {
        final ArgumentHelper argumentHelper = new ArgumentHelper(args, SERVER_TYPE);
        final int serverPort = argumentHelper.getPort() != null ?
                argumentHelper.getPort() :
                getAvailablePortInRange(SERVER_PORT, SERVER_PORT + SERVER_PORT_RANGE);

        // Start Server
        try(final HelloSiLAServer server = new HelloSiLAServer(
                argumentHelper.getInterface(),
                serverPort,
                argumentHelper.getConfigFile().orElse(null))
        ) {
            blockUntilStop();
        }
        System.out.println("termination complete.");
    }

    /**
     * Implementation of the gRPC Service mapped from the "GreetingProvider" Feature, the stubs
     * are automatically generated from the maven plugin.
     */
    static class GreeterImpl extends GreetingProviderGrpc.GreetingProviderImplBase {
        /**
         * Implementation of the SayHello Command, the mapping can be refered to from Part B
         */
        @Override
        public void sayHello(
                GreetingProviderOuterClass.SayHello_Parameters req,
                StreamObserver<GreetingProviderOuterClass.SayHello_Responses> responseObserver
        ) {
            /*
             Different parameters can be checked, it is mandatory to throw Validation Errors in case of
             missing parameters, which will be done automatically in the future.
              */
            if (!req.hasName()) {
                responseObserver.onError(SiLAErrors.generateValidationError(
                        "Name",
                        "Name parameter was not set",
                        "Specify a name with at least one character"));
                return;
            }

            // Custom ValidationError example
            String name = req.getName().getValue();
            if ("error".equalsIgnoreCase(name)) {
                responseObserver.onError(
                        SiLAErrors.generateValidationError(
                                "Name",
                                "Name was called error therefore throw an error :)",
                                "Specify a name that is not \"error\"")
                );
                return;
            }

            // The result of the command is created with the according protobuf builders
            final String msg = "Hello " + name;
            GreetingProviderOuterClass.SayHello_Responses result =
                    GreetingProviderOuterClass.SayHello_Responses
                            .newBuilder()
                            .setGreeting(SiLAFramework.String.newBuilder()
                                    .setValue(msg)
                            )
                            .build();
            responseObserver.onNext(result);
            responseObserver.onCompleted();
            System.out.println("Request received on " + SERVER_TYPE + " " + msg);
        }

        /**
         * Implementation of the unobservable property Start Year
         *
         * It works just like a command but doesn't have any parameters and shouldn't change any
         * state of the server or some underlying instrument or database.
         */
        @Override
        public void getStartYear(
                GreetingProviderOuterClass.Get_StartYear_Parameters request,
                StreamObserver<GreetingProviderOuterClass.Get_StartYear_Responses> responseObserver
        ) {
            responseObserver.onNext(
                    GreetingProviderOuterClass.Get_StartYear_Responses.newBuilder()
                            .setStartYear(
                                    SiLAFramework.Integer.newBuilder()
                                            .setValue(Year.now().getValue())
                                    )
                            .build()
            );
            responseObserver.onCompleted();
        }
    }
}
