package sila_java.servers.multidrop;

import io.grpc.stub.StreamObserver;
import lombok.NonNull;
import lombok.extern.slf4j.Slf4j;
import sila2.org.silastandard.core.dispensecontroller.v1.DispenseControllerGrpc;
import sila2.org.silastandard.core.dispensecontroller.v1.DispenseControllerOuterClass;
import sila2.org.silastandard.core.shakecontroller.v1.ShakeControllerGrpc;
import sila2.org.silastandard.core.shakecontroller.v1.ShakeControllerOuterClass;
import sila_java.library.core.utils.SiLAErrors;
import sila_java.library.server_base.SiLAServerBase;
import sila_java.library.server_base.identification.ServerInformation;
import sila_java.library.server_base.utils.ArgumentHelper;

import javax.annotation.Nullable;
import java.io.Closeable;
import java.io.IOException;
import java.nio.file.Path;
import java.util.HashMap;
import java.util.Map;

import static sila_java.library.core.utils.Utils.blockUntilStop;
import static sila_java.library.core.utils.FileUtils.getResourceContent;
import static sila_java.library.core.utils.SocketUtils.getAvailablePortInRange;
import static sila_java.servers.multidrop.MultidropUtils.MAX_COLUMN;
import static sila_java.servers.multidrop.MultidropUtils.MAX_SHAKE_TIME;
import static sila_java.servers.multidrop.MultidropUtils.MAX_VOLUME;

/**
 * SiLA Server for Multidrop
 *
 * @implNote Hacky Implementation that needs logging and better resource handling
 */
@Slf4j
public class MultidropServer implements Closeable {
    private static final int SERVER_PORT = 50051;
    private static final int SERVER_PORT_RANGE = 256;
    static final String SERVER_TYPE = "MultidropMicro";
    private final MultidropDriver driver;
    private final SiLAServerBase siLAServerBase;

    public MultidropServer(
            @NonNull final String interfaceName,
            final int serverPort,
            @Nullable final Path config) {
        driver = new MultidropDriver();

        try {
            driver.start();
        } catch (IOException e) {
            log.error(e.getMessage());
        }

        try {
            final Map<String, String> fdl = new HashMap<String, String>() {
                {
                    put(
                            "DispenseController",
                            getResourceContent("DispenseController.xml")
                    );
                    put(
                            "ShakeController",
                            getResourceContent("ShakeController.xml")
                    );
                }
            };

            final ServerInformation serverInfo = new ServerInformation(
                    SERVER_TYPE,
                    "Liquid dispenser for 96 and 384 well plates with shaking function. Requires manual "
                            + "installation for a liquid reservoir",
                    "http://www.titertek-berthold.com",
                    "v0.0"
            );

            if (config == null) {
                this.siLAServerBase = SiLAServerBase.withoutConfig(
                        serverInfo,
                        fdl, serverPort, interfaceName,
                        new DispenseImpl(),
                        new ShakeImpl()
                );
            } else {
                this.siLAServerBase = SiLAServerBase.withConfig(
                        config,
                        serverInfo,
                        fdl, serverPort, interfaceName,
                        new DispenseImpl(),
                        new ShakeImpl()
                );
            }
        } catch (IOException e) {
            throw new RuntimeException(e);
        }
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void close() {
        this.siLAServerBase.close();
    }

    public static void main(String[] args) {
        final ArgumentHelper argumentHelper = new ArgumentHelper(args, SERVER_TYPE);
        final int serverPort = argumentHelper.getPort() != null ?
                argumentHelper.getPort() :
                getAvailablePortInRange(SERVER_PORT, SERVER_PORT + SERVER_PORT_RANGE);

        try (final MultidropServer server = new MultidropServer(
                argumentHelper.getInterface(),
                serverPort,
                argumentHelper.getConfigFile().orElse(null))
        ) {
            blockUntilStop();
        }
        log.info("termination complete.");
    }

    class DispenseImpl extends DispenseControllerGrpc.DispenseControllerImplBase {
        @Override
        public void dispensePlate(DispenseControllerOuterClass.DispensePlate_Parameters req,
                                  StreamObserver<DispenseControllerOuterClass.DispensePlate_Responses> responseObserver) {
            genericDispense(req.getVolume().getValue(), 1, 12,
                    responseObserver,
                    DispenseControllerOuterClass.DispensePlate_Responses.newBuilder().build());
        }

        @Override
        public void dispenseColumns(DispenseControllerOuterClass.DispenseColumns_Parameters req,
                                    StreamObserver<DispenseControllerOuterClass.DispenseColumns_Responses> responseObserver) {
            genericDispense(req.getVolume().getValue(),
                    req.getColumnStart().getValue(),
                    req.getColumnEnd().getValue(),
                    responseObserver,
                    DispenseControllerOuterClass.DispenseColumns_Responses.newBuilder().build());
        }

        private <T> void genericDispense(
                long volume, long columnStart, long columndEnd,
                StreamObserver<T> responseObserver, T response){
            if (!MultidropServer.this.driver.isDriverUp()) {
                responseObserver.onError(
                        SiLAErrors.generateExecutionError(
                                "DeviceNotUp",
                                "It seems like the dispenser is not connected or is turned off",
                                "Make sure that the dispenser is turned on and is connected")
                );
                return;
            }

            // @implNote The inputs will be read from the feature definition
            if (volume > MAX_VOLUME) {
                responseObserver.onError(
                        SiLAErrors.generateValidationError(
                                "Volume",
                                "Volume can be maximal " + MAX_VOLUME + " [ml]",
                                "Specify a value less or equal than " + MAX_VOLUME + " [ml]")
                );
                return;
            }

            if (columndEnd > MAX_COLUMN) {
                responseObserver.onError(
                        SiLAErrors.generateValidationError(
                                "ColumnEnd",
                                "Out of bound value",
                                "Specify a value less or equal than " + MAX_COLUMN)
                );
                return;
            }

            try {
                MultidropServer.this.driver.dispense((int) volume, (int) columnStart,
                        (int) columndEnd);
            } catch (IOException e) {
                responseObserver.onError(SiLAErrors.generateExecutionError(
                        "IOException",
                        e.getMessage(),
                        "Check the Multidrop connection"));
                return;
            }

            responseObserver.onNext(response);
            responseObserver.onCompleted();
        }
    }

    class ShakeImpl extends ShakeControllerGrpc.ShakeControllerImplBase {
        @Override
        public void shake(ShakeControllerOuterClass.Shake_Parameters req,
                          StreamObserver<ShakeControllerOuterClass.Shake_Responses> responseObserver){
            if (!MultidropServer.this.driver.isDriverUp()) {
                responseObserver.onError(
                        SiLAErrors.generateExecutionError(
                                "DeviceNotUp",
                                "It seems like the dispenser is not connected or turned off",
                                "")
                );
                return;
            }

            final long duration = req.getDuration().getValue();

            if (duration > MAX_SHAKE_TIME) {
                responseObserver.onError(
                        SiLAErrors.generateValidationError(
                                "Duration",
                                "Maximum duration is " + MAX_SHAKE_TIME + " [s]",
                                "Specify a duration less or equal than " + MAX_SHAKE_TIME + " [s]")
                );
                return;
            }

            try {
                MultidropServer.this.driver.shake((int) duration);
            } catch (IOException e) {
                responseObserver.onError(SiLAErrors.generateExecutionError("IOException", e.getMessage(), ""));
                return;
            }

            responseObserver.onNext(ShakeControllerOuterClass.Shake_Responses.newBuilder().build());
            responseObserver.onCompleted();
        }
    }
}
