package sila_java.servers.test_server;

import lombok.NonNull;
import lombok.extern.slf4j.Slf4j;
import sila_java.library.server_base.SiLAServerBase;
import sila_java.library.server_base.identification.ServerInformation;
import sila_java.library.server_base.utils.ArgumentHelper;
import sila_java.servers.test_server.impl.ObservableProperty;
import sila_java.servers.test_server.impl.UnobservableCommand;
import sila_java.servers.test_server.impl.ObservableCommand;
import sila_java.servers.test_server.impl.UnobservableProperty;

import javax.annotation.Nullable;
import java.io.Closeable;
import java.io.IOException;
import java.nio.file.Path;
import java.util.HashMap;
import java.util.Map;

import static sila_java.library.core.utils.Utils.blockUntilStop;
import static sila_java.library.core.utils.FileUtils.getResourceContent;
import static sila_java.library.core.utils.SocketUtils.getAvailablePortInRange;

/**
 * SiLA Server for test purposes
 */
@Slf4j
public class TestServer implements Closeable {
    static final String SERVER_TYPE = "TestServer";
    private static final int SERVER_PORT = 50053; // Default
    private static final int SERVER_PORT_RANGE = 256;
    private static final String UNOBSERVABLE_PROP_FEATURE = "features/UnobservablePropertyTest.xml";
    private static final String OBSERVABLE_PROP_FEATURE = "features/ObservablePropertyTest.xml";
    private static final String UNOBSERVABLE_CMD_FEATURE = "features/UnobservableCommandTest.xml";
    private static final String OBSERVABLE_CMD_FEATURE = "features/ObservableCommandTest.xml";
    private final SiLAServerBase siLAServerBase;

    public TestServer(
            @NonNull final String interfaceName,
            final int serverPort,
            @Nullable final Path config) {
        try {
            final Map<String, String> featureDefinitions = new HashMap<String, String>() {{
                put("UnobservablePropertyTest", getResourceContent(UNOBSERVABLE_PROP_FEATURE));
                put("ObservablePropertyTest", getResourceContent(OBSERVABLE_PROP_FEATURE));
                put("UnobservableCommandTest", getResourceContent(UNOBSERVABLE_CMD_FEATURE));
                put("ObservableCommandTest", getResourceContent(OBSERVABLE_CMD_FEATURE));
            }};

            final ServerInformation serverInfo = new ServerInformation(
                    SERVER_TYPE,
                    "Server for test purposes",
                    "www.unitelabs.ch",
                    "v0.0"
            );

            if (config == null) {
                this.siLAServerBase = SiLAServerBase.withoutConfig(
                        serverInfo,
                        featureDefinitions,
                        serverPort,
                        interfaceName,
                        new UnobservableProperty(),
                        new ObservableProperty(),
                        new UnobservableCommand(),
                        new ObservableCommand()
                );
            } else {
                this.siLAServerBase = SiLAServerBase.withConfig(
                        config,
                        serverInfo,
                        featureDefinitions,
                        serverPort,
                        interfaceName,
                        new UnobservableProperty(),
                        new ObservableProperty(),
                        new UnobservableCommand(),
                        new ObservableCommand()
                );
            }
        } catch (final IOException e) {
            throw new RuntimeException(e.getMessage());
        }
    }

    @Override
    public void close() {
        this.siLAServerBase.close();
    }

    public static void main(final String[] args) {
        final ArgumentHelper argumentHelper = new ArgumentHelper(args, SERVER_TYPE);
        final int serverPort = argumentHelper.getPort() != null ?
                argumentHelper.getPort() :
                getAvailablePortInRange(SERVER_PORT, SERVER_PORT + SERVER_PORT_RANGE);

        // Start Server
        try (final TestServer server = new TestServer(
                argumentHelper.getInterface(),
                serverPort,
                argumentHelper.getConfigFile().orElse(null))
        ) {
            blockUntilStop();
        }
        System.out.println("Server closed.");
    }
}
