package sila_java.servers.test_server.impl;

import io.grpc.stub.StreamObserver;
import lombok.Getter;
import lombok.NonNull;
import lombok.extern.slf4j.Slf4j;
import sila2.ch.unitelabs.observablecommandtest.v1.ObservableCommandTestGrpc;
import sila2.ch.unitelabs.observablecommandtest.v1.ObservableCommandTestOuterClass;
import sila2.org.silastandard.SiLAFramework;
import sila_java.library.core.utils.SiLAErrors;

import java.util.*;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.TimeUnit;

@Slf4j
public class ObservableCommand extends ObservableCommandTestGrpc.ObservableCommandTestImplBase {
    public static final int RESTART_DELAY = 5; // [sec]
    public static final int UPDATE_DELAY = 10; // [sec]
    public static final ObservableCommandTestOuterClass.RestartDevice_Responses RESTART_RESULT =
            ObservableCommandTestOuterClass.RestartDevice_Responses
            .newBuilder()
            .build();
    public Map<String, Command> commandExecutions = new ConcurrentHashMap<>();

    @Getter
    private static final class Command {
        private final long startTime;
        private final long endTime;
        private final long secondRequired;
        private final long lifeTimeOfExecution;
        private final boolean update;
        private final String id;
        private final SiLAFramework.CommandExecutionUUID executionUUID;

        Command(final boolean update) {
            this.update = update;
            if (this.update)
                this.secondRequired = ObservableCommand.RESTART_DELAY + ObservableCommand.UPDATE_DELAY;
            else
                this.secondRequired = ObservableCommand.RESTART_DELAY;
            this.lifeTimeOfExecution = this.secondRequired * 2;
            this.startTime = System.currentTimeMillis();
            this.endTime = System.currentTimeMillis() + this.secondRequired * 1000;
            this.id = UUID.randomUUID().toString();
            this.executionUUID = SiLAFramework.CommandExecutionUUID
                    .newBuilder()
                    .setCommandId(this.id)
                    .build();
        }

        long getSecondLeft() {
            return (this.endTime - System.currentTimeMillis()) / 1000; // [sec]
        }
    }

    @Override
    public void restartDevice(@NonNull final ObservableCommandTestOuterClass.RestartDevice_Parameters request,
                              @NonNull final StreamObserver<SiLAFramework.CommandConfirmation> responseObserver
    ) {
        final Command c = new Command(request.getUpdateDevice().getValue());
        this.commandExecutions.put(c.getId(), c);
        responseObserver.onNext(SiLAFramework.CommandConfirmation
                .newBuilder()
                .setCommandId(c.getExecutionUUID())
                .setLifetimeOfExecution(SiLAFramework.Duration.newBuilder().setSeconds(c.getLifeTimeOfExecution()).build())
                .build()
        );

        new Timer().schedule(new TimerTask() {
            public void run() {
                log.info("Deleting command execution with id " + c.getId());
                ObservableCommand.this.commandExecutions.remove(c.getId());
            }
        }, TimeUnit.SECONDS.toMillis(c.lifeTimeOfExecution));
        responseObserver.onCompleted();
    }

    @Override
    public void restartDeviceState(@NonNull final SiLAFramework.CommandExecutionUUID request,
                                   @NonNull final StreamObserver<SiLAFramework.ExecutionInfo> responseObserver
    ) {
        final Command c = commandExecutions.get(request.getCommandId());

        new Thread(() -> {
            while (c.getSecondLeft() > 0) {
                responseObserver.onNext(SiLAFramework.ExecutionInfo
                        .newBuilder()
                        .setCommandStatus(SiLAFramework.ExecutionInfo.CommandStatus.running)
                        .setEstimatedRemainingTime(
                                SiLAFramework.Duration.newBuilder().setSeconds(c.getSecondLeft()).build())
                        .setProgressInfo(SiLAFramework.Real.newBuilder().setValue(
                                (double)(c.getSecondRequired() - c.getSecondLeft()) / c.getSecondRequired()).build())
                        .build()
                );
                try {
                    Thread.sleep(1000);
                } catch (InterruptedException e) {
                    e.printStackTrace();
                }
            }
            responseObserver.onNext(SiLAFramework.ExecutionInfo
                            .newBuilder()
                            .setCommandStatus(SiLAFramework.ExecutionInfo.CommandStatus.finishedSuccessfully)
                            .setEstimatedRemainingTime(SiLAFramework.Duration.newBuilder().setSeconds(0).build())
                            .setProgressInfo(SiLAFramework.Real.newBuilder().setValue(1.0d).build())
                            .build());
            responseObserver.onCompleted();
        }).start();
    }

    @Override
    public void restartDeviceResult(@NonNull final SiLAFramework.CommandExecutionUUID request,
                                    @NonNull final StreamObserver<ObservableCommandTestOuterClass.RestartDevice_Responses> responseObserver
    ) {
        if (!this.commandExecutions.containsKey(request.getCommandId())) {
            responseObserver.onError(SiLAErrors.generateExecutionError(
                    "InvalidCommandUUID",
                    "Invalid command UUID",
                    "Verify the specified UUID and make sure that the command UUID is still available")
            );
            return;
        }
        final Command c = this.commandExecutions.get(request.getCommandId());
        if (c.getSecondLeft() > 0) {
            responseObserver.onError(
                    SiLAErrors.generateExecutionError(
                            "RestartInProgress",
                            "Restart is not complete",
                            "Wait a moment and try again")
            );
            return ;
        }
        responseObserver.onNext(ObservableCommand.RESTART_RESULT);
        responseObserver.onCompleted();
    }
}
