/*
 * Decompiled with CFR 0.152.
 */
package org.slinkyframework.environment.docker;

import com.spotify.docker.client.DefaultDockerClient;
import com.spotify.docker.client.DockerClient;
import com.spotify.docker.client.exceptions.DockerCertificateException;
import com.spotify.docker.client.exceptions.DockerException;
import com.spotify.docker.client.messages.Container;
import com.spotify.docker.client.messages.ContainerConfig;
import com.spotify.docker.client.messages.ContainerCreation;
import com.spotify.docker.client.messages.HostConfig;
import com.spotify.docker.client.messages.Image;
import com.spotify.docker.client.messages.PortBinding;
import java.io.Closeable;
import java.io.IOException;
import java.net.Socket;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.function.BiConsumer;
import org.apache.commons.compress.utils.IOUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.slinkyframework.environment.builder.EnvironmentBuilderException;
import org.springframework.retry.RetryPolicy;
import org.springframework.retry.backoff.BackOffPolicy;
import org.springframework.retry.backoff.FixedBackOffPolicy;
import org.springframework.retry.policy.TimeoutRetryPolicy;
import org.springframework.retry.support.RetryTemplate;

public class DockerDriver {
    private static final String ENVIRONMENT_DOCKER_MACHINE_NAME = "DOCKER_MACHINE_NAME";
    private static final String DEFAULT_DOCKER_HOSTNAME = "localhost";
    private static final Logger LOG = LoggerFactory.getLogger(DockerDriver.class);
    private static final int ONE_SECOND = 1000;
    private static final long THIRTY_SECONDS = 30000L;
    private final String containerName;
    private final String imageName;
    private final Map<Integer, Integer> ports;
    private String dockerHostname;
    private DockerClient dockerClient;
    private String containerId;

    public DockerDriver(String containerName, String imageName, Map<Integer, Integer> ports) {
        this.containerName = containerName;
        this.imageName = imageName;
        this.ports = ports;
        this.dockerHostname = System.getenv(ENVIRONMENT_DOCKER_MACHINE_NAME);
        if (this.dockerHostname == null) {
            this.dockerHostname = DEFAULT_DOCKER_HOSTNAME;
        }
        this.connectToDocker();
    }

    private boolean isEnvironmentVariableSet(String name) {
        String value = System.getenv(name);
        return value != null && !value.trim().equals("");
    }

    private void connectToDocker() {
        try {
            LOG.debug("Connecting to Docker");
            this.dockerClient = DefaultDockerClient.fromEnv().build();
            this.dockerClient.ping();
            LOG.debug("Connection to Docker established");
        }
        catch (DockerCertificateException | DockerException | InterruptedException e) {
            throw new EnvironmentBuilderException("Unable to connect to Docker", e);
        }
    }

    public void setUpDocker() {
        LOG.info("Setting up Docker container '{}'", (Object)this.containerName);
        this.pullContainer();
        Optional<Container> existingContainer = this.findExistingContainer();
        if (existingContainer.isPresent()) {
            LOG.warn("Container '{}' already exists", (Object)this.containerName);
            this.killAndRemoveContainer(existingContainer.get());
        }
        ContainerCreation container = this.createContainer();
        this.containerId = container.id();
        this.waitFor(this::startContainer);
        this.waitForContainerToStart();
    }

    private void pullContainer() {
        try {
            if (!this.findImage().isPresent()) {
                this.dockerClient.pull(this.imageName);
            }
        }
        catch (DockerException | InterruptedException e) {
            throw new EnvironmentBuilderException("Unable to pull container: " + this.imageName, e);
        }
    }

    public Optional<Container> findExistingContainer() {
        try {
            List containers = this.dockerClient.listContainers(new DockerClient.ListContainersParam[]{DockerClient.ListContainersParam.allContainers((boolean)true)});
            for (Container container : containers) {
                for (String name : container.names()) {
                    if (!name.contains(this.containerName)) continue;
                    return Optional.of(container);
                }
            }
        }
        catch (DockerException | InterruptedException e) {
            LOG.error("Unable to retrieve a list of Docker containers", e);
        }
        return Optional.empty();
    }

    public void killAndRemoveContainer() {
        Optional<Container> existingContainer = this.findExistingContainer();
        if (existingContainer.isPresent()) {
            this.killAndRemoveContainer(existingContainer.get());
            LOG.info("Docker container '{}' killed and removed", (Object)this.containerName);
        } else {
            LOG.warn("Container '{}' was not running", (Object)this.containerName);
        }
    }

    private void killAndRemoveContainer(Container container) {
        try {
            if (container.status().startsWith("Up")) {
                LOG.debug("Killing Docker container '{}'", (Object)this.containerName);
                this.dockerClient.killContainer(container.id());
            }
            LOG.debug("Removing Docker container '{}'", (Object)this.containerName);
            this.dockerClient.removeContainer(container.id());
        }
        catch (DockerException | InterruptedException e) {
            throw new EnvironmentBuilderException("Unable to kill and remove a container", e);
        }
    }

    private ContainerCreation createContainer() {
        LOG.debug("Creating Docker container '{}'", (Object)this.containerName);
        HashMap portBindings = new HashMap();
        for (int dockerPort : this.ports.keySet()) {
            ArrayList<PortBinding> hostPorts = new ArrayList<PortBinding>();
            hostPorts.add(PortBinding.of((String)"0.0.0.0", (int)this.ports.get(dockerPort)));
            portBindings.put(String.valueOf(dockerPort) + "/tcp", hostPorts);
        }
        HostConfig hostConfig = HostConfig.builder().portBindings(portBindings).build();
        ContainerConfig config = ContainerConfig.builder().image(this.imageName).hostConfig(hostConfig).build();
        try {
            ContainerCreation container = this.dockerClient.createContainer(config, this.containerName);
            LOG.debug("Docker container '{}' created", (Object)this.containerName);
            return container;
        }
        catch (DockerException | InterruptedException e) {
            throw new EnvironmentBuilderException("Unable to create Docker container. Is one already running with the same name?", e);
        }
    }

    private void startContainer(DockerClient docker, String containerId) {
        LOG.info("Starting container '{}", (Object)this.containerName);
        try {
            docker.startContainer(containerId);
            LOG.debug("Container '{}' started", (Object)this.containerName);
        }
        catch (DockerException | InterruptedException e) {
            LOG.error("Unable to start container '{}'. Is there something running on the same ports?", (Object)this.containerName);
            throw new EnvironmentBuilderException("Unable to start container", e);
        }
    }

    private void waitForContainerToStart() {
        TimeoutRetryPolicy retryPolicy = new TimeoutRetryPolicy();
        retryPolicy.setTimeout(30000L);
        FixedBackOffPolicy backOffPolicy = new FixedBackOffPolicy();
        backOffPolicy.setBackOffPeriod(1000L);
        RetryTemplate retryTemplate = new RetryTemplate();
        retryTemplate.setRetryPolicy((RetryPolicy)retryPolicy);
        retryTemplate.setThrowLastExceptionOnExhausted(true);
        retryTemplate.setBackOffPolicy((BackOffPolicy)backOffPolicy);
        retryTemplate.execute(rc -> this.portInUse(this.dockerHostname, this.firstExternalPort()));
    }

    private Integer firstExternalPort() {
        return this.ports.values().toArray(new Integer[0])[0];
    }

    private boolean portInUse(String host, int port) {
        LOG.debug("Check whether {}:{} is ready", (Object)host, (Object)port);
        Socket s = null;
        try {
            s = new Socket(host, port);
            LOG.debug("{}:{} is ready", (Object)host, (Object)port);
        }
        catch (IOException e) {
            try {
                LOG.debug("{}:{} is not ready", (Object)host, (Object)port);
                throw new EnvironmentBuilderException(String.format("Container '%s' has failed to start. Port %s:%s not available", this.containerName, host, port), (Throwable)e);
            }
            catch (Throwable throwable) {
                IOUtils.closeQuietly(s);
                throw throwable;
            }
        }
        IOUtils.closeQuietly((Closeable)s);
        return true;
    }

    public Optional<Image> findImage() {
        try {
            List images = this.dockerClient.listImages(new DockerClient.ListImagesParam[]{DockerClient.ListImagesParam.allImages()});
            for (Image image : images) {
                if (image.repoTags() == null) continue;
                for (String tag : image.repoTags()) {
                    if (!tag.equals(this.imageName)) continue;
                    return Optional.of(image);
                }
            }
        }
        catch (DockerException | InterruptedException e) {
            LOG.error("Unable to retrieve a list of Docker images", e);
        }
        return Optional.empty();
    }

    public void waitFor(BiConsumer<DockerClient, String> function) {
        TimeoutRetryPolicy retryPolicy = new TimeoutRetryPolicy();
        retryPolicy.setTimeout(30000L);
        FixedBackOffPolicy backOffPolicy = new FixedBackOffPolicy();
        backOffPolicy.setBackOffPeriod(1000L);
        RetryTemplate retryTemplate = new RetryTemplate();
        retryTemplate.setRetryPolicy((RetryPolicy)retryPolicy);
        retryTemplate.setThrowLastExceptionOnExhausted(true);
        retryTemplate.setBackOffPolicy((BackOffPolicy)backOffPolicy);
        retryTemplate.execute(rc -> {
            function.accept(this.dockerClient, this.containerId);
            return null;
        });
    }
}

