package org.somda.dsl.biceps

import org.somda.dsl.biceps.base.tree.ComponentTree
import org.somda.dsl.biceps.base.tree.ExtensibleMdibComponent

/**
 * Interface for MDIB validators.
 *
 * Register MDIB validators by using [installValidator].
 */
public interface MdibValidator {
    /**
     * Human-readable name for the validator instance.
     */
    public val name: String

    /**
     * Called before any validation step is executed.
     */
    public fun init() {}

    /**
     * Validator that is executed after the entire tree has been constructed.
     *
     * Use `error(message: String)` to stop MDIB creation and throw an error.
     * In order to increase usability, provide a path to the object where the error occurred by using
     * [org.somda.dsl.biceps.base.tree.ExtensibleMdibComponent.cursor].
     *
     * If not implemented, performs nothing.
     */
    public fun validate(mdib: Mdib, componentTree: ComponentTree.Root) {}

    /**
     * An MDIB component visitor that is called back during tree construction for each node *before* initialization.
     *
     * Use `error(message: String)` to stop MDIB creation and throw an error.
     *
     * The path to the current node is printed to the error output automatically.
     *
     * If not implemented, performs nothing.
     */
    public fun visitPreInit(component: ExtensibleMdibComponent) {}

    /**
     * An MDIB component visitor that is called back during tree construction for each node *after* initialization.
     *
     * Use `error(message: String)` to stop MDIB creation and throw an error.
     *
     * The path to the current node is printed to the error output automatically.
     *
     * If not implemented, performs nothing.
     */
    public fun visitPostInit(component: ExtensibleMdibComponent) {}
}

public fun MdibValidator.queueError(provideMessage: () -> String) {
    currentScope().logError(provideMessage())
}