package org.somda.dsl.biceps.base

import org.somda.dsl.biceps.MdibDsl
import java.math.BigInteger

public class CodingSystem internal constructor(
    private val codingSystem: String,
    private val codingSystemVersion: String? = null
) {
    public infix fun code(code: String): CodedValue {
        return CodedValue(code, codingSystem, codingSystemVersion)
    }

    public infix fun codingSystemVersion(codingSystemVersion: String): CodingSystem {
        return CodingSystem(codingSystem, codingSystemVersion)
    }
}

public fun codingSystem(codingSystem: String): CodingSystem {
    return CodingSystem(codingSystem)
}

public interface CodedValueBase {
    public val code: String
    public val codingSystem: String
    public val codingSystemVersion: String?
}

public typealias CodedValueTranslation = CodedValueBase

public const val DEFAULT_CODING_SYSTEM: String = "urn:oid:1.3.111.2.11073.10101.1"

@MdibDsl
public open class CodedValue(
    public override val code: String,
    public override val codingSystem: String = DEFAULT_CODING_SYSTEM,
    public override val codingSystemVersion: String? = null
) : CodedValueBase {
    public constructor(base: CodedValue): this(base.code, base.codingSystem, base.codingSystemVersion)

    protected val translationList: MutableList<CodedValueTranslation> = mutableListOf()

    protected val conceptDescriptionList: MutableList<LocalizedText> = mutableListOf()

    public fun translation(init: CodedValueTranslation): CodedValueTranslation {
        return init.also {
            translationList.add(it)
        }
    }

    public val translations: List<CodedValueTranslation> = translationList

    public fun conceptDescription(lang: String, value: () -> String = { "" }): LocalizedTextValue {
        return LocalizedTextValue(value(), lang).also {
            conceptDescriptionList.add(it)
        }
    }

    public fun conceptDescriptionRef(
        ref: String,
        version: BigInteger = BigInteger.ZERO,
    ): LocalizedTextReference {
        return LocalizedTextReference(ref, version).also {
            conceptDescriptionList.add(it)
        }
    }

    public fun conceptDescription(text: LocalizedText): LocalizedText {
        return text.also {
            conceptDescriptionList.add(it)
        }
    }

    public val conceptDescriptions: List<LocalizedText> = conceptDescriptionList
}

public fun codedValue(base: CodedValue, init: CodedValue.() -> Unit = {}): CodedValue = CodedValue(base).apply(init)

public fun code(code: String, init: CodedValue.() -> Unit = {}): CodedValue = CodedValue(code).apply(init)

public infix fun CodedValue.codingSystem(codingSystem: String): CodedValue {
    return CodedValue(code, codingSystem, codingSystemVersion)
}

public infix fun CodedValue.code(codingSystem: String): CodedValue {
    return CodedValue(code, codingSystem, codingSystemVersion)
}

public infix fun CodedValue.codingSystemVersion(codingSystemVersion: String): CodedValue {
    return CodedValue(code, codingSystem, codingSystemVersion)
}

public open class IeeePublicCode(code: String) : CodedValue(code.also {
    checkNotNull(code.toUIntOrNull()) {
        "Tried to create a public IEEE code '$it', which is not a valid integer"
    }.also {
        check(isPublicIeee11073CodeId(it)) {
            "Tried to create a public IEEE code '$it' that is outside the public code domain"
        }
    }
}) {
    public constructor(base: CodedValue): this(base.code)
}

public open class IeeePrivateCode(code: String) : CodedValue(code.also {
    checkNotNull(code.toUIntOrNull()) {
        "Tried to create a private IEEE code '$it', which is not a valid integer"
    }.also {
        check(!isPublicIeee11073CodeId(it)) {
            "Tried to create a private IEEE code '$it' that is outside the private code domain"
        }
    }
}) {
    public constructor(base: CodedValue): this(base.code)
}

// ieee private partition
private const val MDC_PART_PVT = 1024u
private val startOfPrivatePartition = MDC_PART_PVT.shl(16)
private val endOfPrivatePartition = (((MDC_PART_PVT + 1u).shl(16)) - 1u)

private fun isPublicIeee11073CodeId(code: UInt): Boolean {
    // 1. Check for private partition (MDC_PART_PVT = 1024)
    return if ((code >= startOfPrivatePartition) && (code <= endOfPrivatePartition)) {
        false
    } else {
        // 2. Check for private range in partitions
        // In each partition 4096 entries in range 0xF000 - 0xFFFF (61440 -
        // 65535).
        code.and(0xF000u) != 0xF000u
    }
}