package org.somda.dsl.biceps.base.tree

import org.somda.dsl.biceps.Mdib
import org.somda.dsl.biceps.alert.AlertCondition
import org.somda.dsl.biceps.alert.AlertSystem
import org.somda.dsl.biceps.alert.LimitAlertCondition
import org.somda.dsl.biceps.base.tree.ComponentTree.Root
import org.somda.dsl.biceps.component.Channel
import org.somda.dsl.biceps.component.Mds
import org.somda.dsl.biceps.component.Vmd
import org.somda.dsl.biceps.context.SystemContext
import org.somda.dsl.biceps.sco.Sco

public sealed interface ComponentTree {
    public val children: List<Node<*>>

    public class Root : ComponentTree {
        public override lateinit var children: List<Node<*>>
            internal set

        public fun traversePreOrder(visitor: (Node<*>) -> Unit) {
            children.forEach {
                visitor(it)
                it.traversePreOrder(visitor)
            }
        }
    }

    public class Node<V : IdentifiableMdibComponent>(
        public val component: V,
    ) : ComponentTree {
        public override lateinit var children: List<Node<*>>
            internal set

        public lateinit var parent: ComponentTree
            internal set

        public inline fun <reified T : IdentifiableMdibComponent> withComponent(action: (T) -> Unit) {
            (component as? T)?.let { action(it) }
        }

        public inline fun <reified T : IdentifiableMdibComponent> withTyped(action: (Node<T>) -> Unit): Unit {
            (component as? T)?.also {
                // suppress is ok as component was tested non-null before
                @Suppress("UNCHECKED_CAST")
                action(this as Node<T>)
            }
        }

        public inline fun <reified T : IdentifiableMdibComponent> forDirectChildren(action: (List<Node<T>>) -> Unit) {
            action(directChildren())
        }

        public inline fun <reified T : IdentifiableMdibComponent> directChildren(): List<Node<T>> {
            return children.filter { it.component as? T != null }.mapNotNull {
                // suppress is ok as component was tested non-null before
                @Suppress("UNCHECKED_CAST")
                it as? Node<T>
            }
        }

        public inline fun <reified T : IdentifiableMdibComponent> forAnyChildren(action: (List<Node<T>>) -> Unit) {
            action(anyChildren())
        }

        public inline fun <reified T : IdentifiableMdibComponent> anyChildren(): List<Node<T>> {
            return fullClosure().filter { it.component as? T != null }.map {
                // suppress is ok as component was tested non-null before
                @Suppress("UNCHECKED_CAST")
                it as Node<T>
            }
        }

        public inline fun <reified T : IdentifiableMdibComponent> forFirstChild(action: (Node<T>) -> Unit) {
            children.firstOrNull { it.component as? T != null }?.also {
                // suppress is ok as component was tested non-null before
                @Suppress("UNCHECKED_CAST")
                action(it as Node<T>)
            }
        }

        public fun fullClosure(): List<Node<*>> {
            return fullClosure(this)
        }

        private fun fullClosure(node: Node<*>): List<Node<*>> {
            return mutableListOf<Node<*>>().apply {
                add(node)
                node.children.forEach { addAll(fullClosure(it)) }
            }
        }

        override fun toString(): String {
            return "Node($component)"
        }

        public fun traversePreOrder(visitor: (Node<*>) -> Unit) {
            visitor(this)
            this.children.forEach {
                it.traversePreOrder(visitor)
            }
        }
    }
}

internal fun buildComponentTree(mdib: Mdib): Root {
    return Root().also { root ->
        root.children = mdib.mdss.map {
            nodeFor(it, root)
        }
    }
}

private fun buildChildren(
    node: ComponentTree.Node<*>
): List<ComponentTree.Node<*>> {
    return mutableListOf<ComponentTree.Node<*>>().apply {
        when (val component = node.component) {
            is Mds -> {
                component.clock?.also { add(nodeFor(it, node)) }
                component.batteries.forEach { add(nodeFor(it, node)) }
                component.systemContext?.also { add(nodeFor(it, node)) }
                component.alertSystem?.also { add(nodeFor(it, node)) }
                component.sco?.also { add(nodeFor(it, node)) }
                component.vmds.forEach { add(nodeFor(it, node)) }
            }

            is Vmd -> {
                component.alertSystem?.also { add(nodeFor(it, node)) }
                component.sco?.also { add(nodeFor(it, node)) }
                component.channels.forEach { add(nodeFor(it, node)) }
            }

            is SystemContext -> {
                component.patientContext?.also { add(nodeFor(it, node)) }
                component.locationContext?.also { add(nodeFor(it, node)) }
                component.ensembleContexts.forEach { add(nodeFor(it, node)) }
            }

            is AlertSystem -> {
                component.alertSignals.forEach { add(nodeFor(it, node)) }
                component.alertConditions.forEach {
                    when (it) {
                        is AlertCondition -> add(nodeFor(it, node))
                        is LimitAlertCondition -> add(nodeFor(it, node))
                    }
                }
            }

            is Channel -> {
                component.metrics.forEach { add(nodeFor(it, node)) }
            }

            is Sco -> {
                component.operations.forEach { add(nodeFor(it, node)) }
            }

            else -> Unit
        }
    }
}

private fun nodeFor(
    component: IdentifiableMdibComponent,
    parent: ComponentTree
): ComponentTree.Node<*> {
    return ComponentTree.Node(component).also { node ->
        node.parent = parent
        node.children = buildChildren(node)
    }
}