package org.somda.dsl.biceps.base.tree

import org.somda.dsl.biceps.MdibDsl
import org.somda.dsl.biceps.base.CodedValue
import org.somda.dsl.biceps.base.Handle
import org.somda.dsl.biceps.base.SafetyClassification
import org.somda.dsl.biceps.checkUnset
import org.somda.dsl.biceps.currentScope

public typealias StringProperties = Map<String, String?>

private fun logProgress(str: String) {
    currentScope().also { scope ->
        scope.cursor().toString().also {
            val prefix = if (it.isNotEmpty()) "$it: " else ""
            scope.logger.info { "${prefix}Add $str" }
        }
    }
}

private inline fun <reified T : Any> T.stringify(handle: Handle, properties: Map<String, String?>): String {
    val props = properties.let { map ->
        if (map.entries.isNotEmpty()) {
            map.entries.joinToString(", ", ", ") { entry ->
                "${entry.key}=${entry.value?.let { "'$it'" } ?: "<none>"}"
            }
        } else {
            ""
        }
    }

    return "${this::class.simpleName}('${handle.name}'$props)"
}

@MdibDsl
public abstract class Descriptor<T : Any>(
    handle: Handle,
    private val newState: () -> T,
    additionalToStringProperties: StringProperties = emptyMap()
) : IdentifiableMdibComponent(handle) {
    private val stringRepresentation = stringify(handle, additionalToStringProperties)

    public var state: T? = null
        private set

    public var type: CodedValue? = null
        private set

    public var safetyClassification: SafetyClassification? = null
        private set

    public fun state(init: T.() -> Unit = {}): T = newState().apply(init).also {
        state = it
    }

    public fun type(base: CodedValue, init: CodedValue.() -> Unit = {}): CodedValue =
        CodedValue(base).apply(init).also {
            checkUnset(type, "type") {
                type = it
            }
        }

    public fun safetyClassification(init: SafetyClassification): SafetyClassification = init.also {
        checkUnset(safetyClassification, "safetyClassification") {
            safetyClassification = it
        }
    }

    public override fun toString(): String = stringRepresentation


    init {
        logProgress(stringRepresentation)
    }
}

@MdibDsl
public abstract class DescriptorMultiState<T : Any>(
    handle: Handle,
    private val newState: () -> T,
    additionalToStringProperties: StringProperties = emptyMap()
) : IdentifiableMdibComponent(handle) {
    private val stringRepresentation = stringify(handle, additionalToStringProperties)

    // todo there is currently no support to set context states in the dsl
    private val stateList: MutableList<T> = mutableListOf()

    public var type: CodedValue? = null
        private set

    // set public to enable
    private fun state(init: T.() -> Unit): T = newState().apply(init).also {
        stateList.add(it)
    }

    public val states: List<T> = stateList

    public var safetyClassification: SafetyClassification? = null
        private set

    public fun type(base: CodedValue, init: CodedValue.() -> Unit = {}): CodedValue =
        CodedValue(base).apply(init).also {
            checkUnset(type, "type") {
                type = it
            }
        }

    public fun safetyClassification(init: SafetyClassification): SafetyClassification = init.also {
        checkUnset(safetyClassification, "safetyClassification") {
            safetyClassification = it
        }
    }

    public override fun toString(): String = stringRepresentation

    init {
        logProgress(stringRepresentation)
    }
}