package org.somda.dsl.rendering.jaxb

import org.somda.dsl.biceps.Mdib
import org.somda.dsl.biceps.component.Battery
import org.somda.dsl.biceps.component.Channel
import org.somda.dsl.biceps.component.Clock
import org.somda.dsl.biceps.component.Mds
import org.somda.dsl.biceps.component.Vmd
import org.somda.dsl.biceps.context.SystemContext
import org.somda.dsl.rendering.jaxb.mapping.AlertMapping
import org.somda.dsl.rendering.jaxb.mapping.BaseMapping
import org.somda.dsl.rendering.jaxb.mapping.ContextMapping
import org.somda.dsl.rendering.jaxb.mapping.MetricMapping
import org.somda.dsl.rendering.jaxb.mapping.OperationMapping
import org.somda.dsl.rendering.jaxb.mapping.base.mapCodedValue
import org.somda.dsl.rendering.jaxb.mapping.base.mapMdsOperatingMode
import org.somda.dsl.rendering.jaxb.mapping.base.mapMeasuremnt
import org.somda.dsl.rendering.jaxb.mapping.base.mapPhysicalConnectorInfo
import org.somda.sdc.biceps.model.message.GetMdibResponse
import org.somda.sdc.biceps.model.message.ObjectFactory
import org.somda.sdc.biceps.model.participant.AbstractState
import org.somda.sdc.biceps.model.participant.BatteryDescriptor
import org.somda.sdc.biceps.model.participant.BatteryState
import org.somda.sdc.biceps.model.participant.ChannelDescriptor
import org.somda.sdc.biceps.model.participant.ChannelState
import org.somda.sdc.biceps.model.participant.ClockDescriptor
import org.somda.sdc.biceps.model.participant.ClockState
import org.somda.sdc.biceps.model.participant.MdState
import org.somda.sdc.biceps.model.participant.MdsDescriptor
import org.somda.sdc.biceps.model.participant.MdsState
import org.somda.sdc.biceps.model.participant.SystemContextDescriptor
import org.somda.sdc.biceps.model.participant.SystemContextState
import org.somda.sdc.biceps.model.participant.VmdDescriptor
import org.somda.sdc.biceps.model.participant.VmdState
import java.io.OutputStream
import java.util.*
import kotlin.time.toJavaDuration


internal class MdibRenderer(
    extensions: Set<ExtensionRenderer> = setOf(),
    private val dslMdib: Mdib
) {
    private val bicepsContexts = listOf(
        JaxbContext("org.somda.sdc.biceps.model.extension"),
        JaxbContext("org.somda.sdc.biceps.model.participant"),
        JaxbContext("org.somda.sdc.biceps.model.message")
    )

    private val bicepsSchemaPaths = listOf(
        XmlSchemaPath("ExtensionPoint.xsd"),
        XmlSchemaPath("BICEPS_ParticipantModel.xsd"),
        XmlSchemaPath("BICEPS_MessageModel.xsd")
    )

    private val bicepsPrefixes = mapOf(
        Namespace("http://standards.ieee.org/downloads/11073/11073-10207-2017/extension") to NamespacePrefix("ext"),
        Namespace("http://standards.ieee.org/downloads/11073/11073-10207-2017/participant") to NamespacePrefix("pm"),
        Namespace("http://standards.ieee.org/downloads/11073/11073-10207-2017/message") to NamespacePrefix("msg")
    )

    private val jaxbMarshaller = DslJaxbMarshalling(
        bicepsContexts + extensions.mapNotNull { it.jaxbContextPath },
        bicepsSchemaPaths + extensions.mapNotNull { it.xmlSchemaPath },
        bicepsPrefixes + extensions.mapNotNull { it.namespaceMapping }.associate { it.second to it.first }
    )

    private val mappedStates = mutableListOf<AbstractState>()
    private val alertMapping = AlertMapping(mappedStates)
    private val operationMapping = OperationMapping(mappedStates)
    private val baseMapping = BaseMapping(mappedStates, alertMapping::mapAlertSystem, operationMapping::mapOperation)
    private val metricMapping = MetricMapping(mappedStates)
    private val contextMapping = ContextMapping(mappedStates)


    public fun renderToGetMdibResponse(): GetMdibResponse {
        return ObjectFactory().createGetMdibResponse().apply {
            mapMdib().also {
                this.sequenceId = it.sequenceId
                this.mdib = it
            }
        }
    }

    public fun renderTo(outputStream: OutputStream) {
        renderToGetMdibResponse().also {
            jaxbMarshaller.marshal(it, outputStream)
        }
    }

    private fun mapMdib() = org.somda.sdc.biceps.model.participant.Mdib().apply {
        sequenceId = UUID.randomUUID().toString().let { "urn:uuid:$it" }
        dslMdib.mdibVersion?.also {
            sequenceId = it.sequenceId
            instanceId = it.instanceId
            mdibVersion = it.version
        }

        mappedStates.clear()
        mdDescription = mapMdDescription()

        mdState = MdState().apply {
            state = mappedStates
        }
    }

    private fun mapMdDescription() = org.somda.sdc.biceps.model.participant.MdDescription().apply {
        for (mdsChild in dslMdib.mdss) {
            mds.add(mapMds(mdsChild))
        }
    }

    private fun mapMds(src: Mds) = MdsDescriptor().apply {
        // todo map MetaData

        baseMapping.mapIntoAbstractComplexDeviceComponentDescriptor(src, this)
        systemContext = mapSystemContext(src.systemContext)

        for (dslVmd in src.vmds) {
            vmd.add(mapVmd(dslVmd))
        }

        clock = mapClock(src.clock)
        battery = src.batteries.map { mapBattery(it) }

        mappedStates.add(mapMdsState(src))
    }

    private fun mapBattery(src: Battery?) = src?.let {
        BatteryDescriptor().apply {
            baseMapping.mapIntoAbstractDeviceComponentDescriptor(src, this)

            voltageSpecified = mapMeasuremnt(src.voltageSpecified)
            capacitySpecified = mapMeasuremnt(src.capacitySpecified)
            capacityFullCharge = mapMeasuremnt(src.capacityFullCharge)

            mappedStates.add(mapBatteryState(src))
        }
    }

    private fun mapBatteryState(src: Battery) = BatteryState().apply {
        baseMapping.mapIntoAbstractDeviceComponentState(src.state, src.handle.ref, src.version, this)
        src.state?.also {
            physicalConnector = mapPhysicalConnectorInfo(it.physicalConnector)
        }
    }

    private fun mapClock(src: Clock?) = src?.let {
        ClockDescriptor().apply {
            baseMapping.mapIntoAbstractDeviceComponentDescriptor(src, this)

            resolution = src.resolution?.toJavaDuration()
            timeProtocol = src.timeProtocols.map { mapCodedValue(it) }

            mappedStates.add(mapClockState(src))
        }
    }

    private fun mapClockState(src: Clock) = ClockState().apply {
        baseMapping.mapIntoAbstractDeviceComponentState(src.state, src.handle.ref, src.version, this)
        src.state?.also {
            accuracy = it.accuracy?.value
            physicalConnector = mapPhysicalConnectorInfo(it.physicalConnector)
            activeSyncProtocol = mapCodedValue(it.activeSyncProtocol)
            timeZone = it.timeZone?.value
            isCriticalUse = it.isCriticalUse()
            isRemoteSync = it.isRemoteSynchronization()
            referenceSource = it.referenceSources
        }
    }

    private fun mapMdsState(src: Mds) = MdsState().apply {
        baseMapping.mapIntoAbstractDeviceComponentState(src.state, src.handle.ref, src.version, this)
        src.state?.also {
            operatingMode = mapMdsOperatingMode(it.mdsOperatingMode)
            physicalConnector = mapPhysicalConnectorInfo(it.physicalConnector)
            lang = it.lang?.toLanguageTag()
        }
    }

    private fun mapSystemContext(src: SystemContext?): SystemContextDescriptor? {
        return src?.let {
            SystemContextDescriptor().apply {
                baseMapping.mapIntoAbstractDeviceComponentDescriptor(src, this)
                this.patientContext = src.patientContext?.let { contextMapping.mapPatientContext(it) }
                this.locationContext = src.locationContext?.let { contextMapping.mapLocationContext(it) }
                for (ensembleContext in src.ensembleContexts) {
                    this.ensembleContext.add(contextMapping.mapEnsembleContext(ensembleContext))
                }

                mappedStates.add(mapSystemContextState(src))
            }
        }
    }

    private fun mapSystemContextState(
        src: SystemContext
    ): SystemContextState {
        return SystemContextState().apply {
            baseMapping.mapIntoAbstractDeviceComponentState(src.state, src.handle.ref, src.version, this)
        }
    }

    private fun mapVmd(src: Vmd) = VmdDescriptor().apply {
        baseMapping.mapIntoAbstractComplexDeviceComponentDescriptor(src, this)
        for (dslChannel in src.channels) {
            channel.add(mapChannel(dslChannel))
        }

        mappedStates.add(mapVmdState(src))
    }

    private fun mapVmdState(src: Vmd) = VmdState().apply {
        baseMapping.mapIntoAbstractDeviceComponentState(src.state, src.handle.ref, src.version, this)

    }

    private fun mapChannel(src: Channel) = ChannelDescriptor().apply {
        baseMapping.mapIntoAbstractDeviceComponentDescriptor(src, this)

        for (dslMetric in src.metrics) {
            metric.add(metricMapping.mapMetric(dslMetric))
        }

        mappedStates.add(mapChannelState(src))
    }

    private fun mapChannelState(src: Channel): ChannelState {
        return ChannelState().apply {
            baseMapping.mapIntoAbstractDeviceComponentState(src.state, src.handle.ref, src.version, this)
        }
    }
}
