package org.somda.dsl.rendering.jaxb.mapping

import org.somda.dsl.biceps.alert.AlertCondition
import org.somda.dsl.biceps.alert.AlertConditionBase
import org.somda.dsl.biceps.alert.AlertSignal
import org.somda.dsl.biceps.alert.AlertSystem
import org.somda.dsl.biceps.alert.LimitAlertCondition
import org.somda.dsl.biceps.base.range
import org.somda.dsl.rendering.jaxb.mapping.base.mapAlertConditionKind
import org.somda.dsl.rendering.jaxb.mapping.base.mapAlertConditionMonitoredLimits
import org.somda.dsl.rendering.jaxb.mapping.base.mapAlertConditionPriority
import org.somda.dsl.rendering.jaxb.mapping.base.mapAlertSignalPresence
import org.somda.dsl.rendering.jaxb.mapping.base.mapAlertSignalPrimaryLocation
import org.somda.dsl.rendering.jaxb.mapping.base.mapCodedValue
import org.somda.dsl.rendering.jaxb.mapping.base.mapMapAlertSignalManifestation
import org.somda.dsl.rendering.jaxb.mapping.base.mapRange

import org.somda.sdc.biceps.model.participant.AbstractState
import org.somda.sdc.biceps.model.participant.AlertConditionDescriptor
import org.somda.sdc.biceps.model.participant.AlertConditionMonitoredLimits
import org.somda.sdc.biceps.model.participant.AlertConditionState
import org.somda.sdc.biceps.model.participant.AlertSignalDescriptor
import org.somda.sdc.biceps.model.participant.AlertSignalState
import org.somda.sdc.biceps.model.participant.AlertSystemDescriptor
import org.somda.sdc.biceps.model.participant.AlertSystemState
import org.somda.sdc.biceps.model.participant.LimitAlertConditionDescriptor
import org.somda.sdc.biceps.model.participant.LimitAlertConditionState
import kotlin.time.toJavaDuration

internal class AlertMapping(private val mappedStates: MutableList<AbstractState>) {
    private val baseMapping = BaseMapping(mappedStates, ::mapAlertSystem)

    fun mapAlertSystem(src: AlertSystem?): AlertSystemDescriptor? {
        return src?.let {
            AlertSystemDescriptor().apply {
                baseMapping.mapIntoAbstractDescriptor(src, this)
                this.selfCheckPeriod = it.selfCheckPeriod?.toJavaDuration()
                this.alertCondition = it.alertConditions.map(::mapAlertConditionBase)
                this.alertSignal = it.alertSignals.map(::mapAlertSignal)
                this.maxTechnicalParallelAlarms = it.maxTechnicalParallelAlarms
                this.maxPhysiologicalParallelAlarms = it.maxPhysiologicalParallelAlarms

                mappedStates.add(mapAlertSystemState(src))
            }
        }
    }

    private fun mapAlertSystemState(src: AlertSystem): AlertSystemState {
        return AlertSystemState().apply {
            baseMapping.mapIntoAbstractAlertState(src.state, src.handle.ref, src.version, this)
        }
    }

    private fun mapAlertConditionBase(src: AlertConditionBase): AlertConditionDescriptor {
        return when (src) {
            is AlertCondition -> mapAlertCondition(src)
            is LimitAlertCondition -> mapLimitAlertCondition(src)
        }
    }

    private fun mapAlertCondition(src: AlertCondition) = AlertConditionDescriptor().apply {
        baseMapping.mapIntoAbstractDescriptor(src, this)
        this.kind = mapAlertConditionKind(src.kind)
        this.type = mapCodedValue(src.type)
        this.source = src.sources.map { it.name }
        this.canDeescalate = mapAlertConditionPriority(src.canDeescalate)
        this.canEscalate = mapAlertConditionPriority(src.canEscalate)
        this.priority = mapAlertConditionPriority(src.priority)
        this.defaultConditionGenerationDelay = src.defaultConditionGenerationDelay?.toJavaDuration()

        mappedStates.add(mapAlertConditionState(src))
    }

    private fun mapAlertConditionState(src: AlertCondition): AlertConditionState {
        return AlertConditionState().apply {
            baseMapping.mapIntoAbstractAlertState(src.state, src.handle.ref, src.version, this)
            src.state?.also {
                rank = it.rank
                actualPriority = mapAlertConditionPriority(it.actualPriority)
                actualConditionGenerationDelay = it.actualConditionGenerationDelay?.toJavaDuration()
                isPresence = it.presence
            }
        }
    }

    private fun mapLimitAlertCondition(src: LimitAlertCondition) = LimitAlertConditionDescriptor().apply {
        baseMapping.mapIntoAbstractDescriptor(src, this)
        this.kind = mapAlertConditionKind(src.kind)
        this.type = mapCodedValue(src.type)
        this.source = src.sources.map { it.name }
        this.canDeescalate = mapAlertConditionPriority(src.canDeescalate)
        this.canEscalate = mapAlertConditionPriority(src.canEscalate)
        this.maxLimits = mapRange(src.maxLimits)
        this.isAutoLimitSupported = src.isAutoLimitSupported()
        this.defaultConditionGenerationDelay = src.defaultConditionGenerationDelay?.toJavaDuration()
        this.priority = mapAlertConditionPriority(src.priority)

        mappedStates.add(mapLimitAlertConditionState(src))
    }

    private fun mapLimitAlertConditionState(src: LimitAlertCondition): LimitAlertConditionState {
        return LimitAlertConditionState().apply {
            baseMapping.mapIntoAbstractAlertState(src.state, src.handle.ref, src.version, this)
            monitoredAlertLimits = mapAlertConditionMonitoredLimits(src.state?.monitoredAlertLimits)
                ?: AlertConditionMonitoredLimits.NONE
            limits = mapRange(src.state?.limits ?: range())
            src.state?.also {
                rank = it.rank
                actualPriority = mapAlertConditionPriority(it.actualPriority)
                actualConditionGenerationDelay = it.actualConditionGenerationDelay?.toJavaDuration()
            }
        }
    }

    private fun mapAlertSignal(src: AlertSignal) = AlertSignalDescriptor().apply {
        baseMapping.mapIntoAbstractDescriptor(src, this)
        this.conditionSignaled = src.conditionSignaled?.name
        this.isLatching = src.isLatching()
        this.acknowledgeTimeout = src.acknowledgeTimeout?.toJavaDuration()
        this.defaultSignalGenerationDelay = src.defaultSignalGenerationDelay?.toJavaDuration()
        this.isAcknowledgementSupported = src.isAcknowledgementSupported()
        this.manifestation = mapMapAlertSignalManifestation(src.manifestation)

        mappedStates.add(mapAlertSignalState(src))
    }

    private fun mapAlertSignalState(src: AlertSignal): AlertSignalState {
        return AlertSignalState().apply {
            baseMapping.mapIntoAbstractAlertState(src.state, src.handle.ref, src.version, this)
            src.state?.also {
                slot = it.slot
                location = mapAlertSignalPrimaryLocation(it.location)
                actualSignalGenerationDelay = it.actualSignalGenerationDelay?.toJavaDuration()
                presence = mapAlertSignalPresence(it.presence)
            }
        }
    }
}